# PXEBot Deployment Guide

## Pre-Deployment Checklist

- [ ] Google Cloud Project created with YouTube Data API v3 enabled
- [ ] OAuth2 credentials configured (Client ID, Secret, Redirect URI)
- [ ] OpenAI API key obtained
- [ ] Production domain configured with SSL
- [ ] MySQL 8 and Redis available
- [ ] Supervisor or Docker configured for Horizon

---

## Local Development Setup

### Using Docker (Recommended)

```bash
# 1. Clone repository
git clone https://github.com/yourorg/pxebot.git
cd pxebot

# 2. Copy environment file
cp .env.example .env

# 3. Configure .env
nano .env
# Set: DB_*, YT_CLIENT_ID, YT_CLIENT_SECRET, OPENAI_API_KEY

# 4. Start Docker services
docker-compose up -d

# 5. Install dependencies
docker-compose exec app composer install
docker-compose exec app npm install

# 6. Generate app key
docker-compose exec app php artisan key:generate

# 7. Run migrations
docker-compose exec app php artisan migrate --seed

# 8. Link storage
docker-compose exec app php artisan storage:link

# 9. Access application
# App: http://localhost
# Horizon: http://localhost/horizon
```

### Using Local PHP/MySQL

```bash
# 1. Install dependencies
composer install
npm install

# 2. Configure .env
cp .env.example .env
php artisan key:generate

# 3. Start services
# MySQL: service mysql start
# Redis: service redis-server start

# 4. Migrate database
php artisan migrate --seed

# 5. Start queue worker
php artisan horizon

# 6. Start scheduler (separate terminal)
php artisan schedule:work

# 7. Start dev server
php artisan serve
```

---

## Production Deployment

### Option 1: Docker on VPS

#### Server Requirements
- Ubuntu 20.04+
- 2GB RAM minimum
- 20GB storage
- Docker & Docker Compose installed

#### Deployment Steps

```bash
# 1. Clone repository on server
ssh user@your-server.com
cd /var/www
git clone https://github.com/yourorg/pxebot.git
cd pxebot

# 2. Configure production environment
cp .env.example .env
nano .env
# Set APP_ENV=production
# Set APP_DEBUG=false
# Set APP_URL=https://your-domain.com
# Configure all production credentials

# 3. Build production Docker image
docker-compose build --no-cache

# 4. Start services
docker-compose up -d

# 5. Install dependencies
docker-compose exec app composer install --optimize-autoloader --no-dev
docker-compose exec app npm install --production
docker-compose exec app npm run build

# 6. Run migrations
docker-compose exec app php artisan migrate --force

# 7. Cache configuration
docker-compose exec app php artisan config:cache
docker-compose exec app php artisan route:cache
docker-compose exec app php artisan view:cache

# 8. Set permissions
docker-compose exec app chown -R www-data:www-data storage bootstrap/cache
```

#### Setup Cron for Scheduler

```bash
# Edit crontab
crontab -e

# Add this line
* * * * * cd /var/www/pxebot && docker-compose exec -T app php artisan schedule:run >> /dev/null 2>&1
```

#### Setup Nginx Reverse Proxy (if needed)

```nginx
server {
    listen 80;
    server_name your-domain.com;

    location / {
        return 301 https://$server_name$request_uri;
    }
}

server {
    listen 443 ssl http2;
    server_name your-domain.com;

    ssl_certificate /etc/letsencrypt/live/your-domain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/your-domain.com/privkey.pem;

    location / {
        proxy_pass http://localhost:80;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
```

#### SSL with Let's Encrypt

```bash
sudo apt install certbot python3-certbot-nginx
sudo certbot --nginx -d your-domain.com -d www.your-domain.com
```

### Option 2: Traditional VPS (No Docker)

```bash
# 1. Install PHP 8.2+
sudo apt update
sudo apt install php8.2 php8.2-fpm php8.2-mysql php8.2-redis php8.2-mbstring \
    php8.2-xml php8.2-bcmath php8.2-curl php8.2-zip php8.2-gd

# 2. Install Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer

# 3. Clone and setup
cd /var/www
git clone https://github.com/yourorg/pxebot.git
cd pxebot
composer install --optimize-autoloader --no-dev

# 4. Configure .env
cp .env.example .env
nano .env
php artisan key:generate

# 5. Migrate
php artisan migrate --force

# 6. Setup Supervisor for Horizon
sudo apt install supervisor
sudo nano /etc/supervisor/conf.d/pxebot-horizon.conf
```

#### Supervisor Config (`/etc/supervisor/conf.d/pxebot-horizon.conf`)

```ini
[program:pxebot-horizon]
process_name=%(program_name)s
command=php /var/www/pxebot/artisan horizon
autostart=true
autorestart=true
user=www-data
redirect_stderr=true
stdout_logfile=/var/www/pxebot/storage/logs/horizon.log
stopwaitsecs=3600
```

```bash
# Reload supervisor
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start pxebot-horizon:*
```

#### Cron for Scheduler

```bash
crontab -e
# Add:
* * * * * cd /var/www/pxebot && php artisan schedule:run >> /dev/null 2>&1
```

#### Nginx Config

```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /var/www/pxebot/public;

    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-Content-Type-Options "nosniff";

    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }
}
```

---

## Post-Deployment Tasks

### 1. Verify Services

```bash
# Check Docker containers
docker-compose ps

# Check Horizon
curl http://localhost/horizon

# Check health endpoint
curl http://localhost/health
```

### 2. Create Admin User

```bash
# SSH into server
docker-compose exec app php artisan tinker

# In tinker:
$user = App\Models\User::create([
    'name' => 'Admin',
    'email' => 'admin@phxbot.com',
    'password' => bcrypt('secure-password'),
    'is_admin' => true,
    'email_verified_at' => now()
]);
```

### 3. Monitor Logs

```bash
# Application logs
tail -f storage/logs/laravel.log

# Horizon logs
tail -f storage/logs/horizon.log

# Nginx logs
tail -f /var/log/nginx/error.log
```

### 4. Setup Monitoring (Optional)

Consider adding:
- **Laravel Telescope** (already included) for local debugging
- **Sentry** for error tracking
- **New Relic** or **DataDog** for APM
- **Prometheus** + **Grafana** for metrics

---

## Backup Strategy

### Database Backup

```bash
# Automated daily backup script
#!/bin/bash
TIMESTAMP=$(date +%F_%H-%M-%S)
BACKUP_DIR="/backups/pxebot"
mkdir -p $BACKUP_DIR

docker-compose exec -T mysql mysqldump -u laravel -psecret pxebot | gzip > $BACKUP_DIR/pxebot_$TIMESTAMP.sql.gz

# Keep only last 7 days
find $BACKUP_DIR -name "*.sql.gz" -mtime +7 -delete
```

Add to crontab:
```
0 2 * * * /path/to/backup-script.sh
```

---

## Scaling & Performance

### Horizontal Scaling

1. **Load Balancer**: Use Nginx or HAProxy to distribute traffic
2. **Multiple Workers**: Run multiple Horizon workers on different servers
3. **Redis Cluster**: Use Redis Cluster for distributed caching
4. **Database Replication**: Master-slave MySQL setup

### Optimization

```bash
# Enable OPcache (php.ini)
opcache.enable=1
opcache.memory_consumption=256
opcache.max_accelerated_files=20000

# Queue optimization
php artisan horizon:publish
# Edit config/horizon.php to increase maxProcesses

# Database indexes
# All critical indexes are in migrations
```

---

## Troubleshooting

### Horizon Not Processing Jobs

```bash
# Check Horizon status
docker-compose exec app php artisan horizon:status

# Restart Horizon
docker-compose restart horizon
# OR
sudo supervisorctl restart pxebot-horizon:*
```

### Permission Issues

```bash
docker-compose exec app chown -R www-data:www-data storage bootstrap/cache
docker-compose exec app chmod -R 775 storage bootstrap/cache
```

### OAuth Redirect Mismatch

Ensure `YT_REDIRECT_URI` in `.env` matches exactly what's configured in Google Cloud Console.

### Rate Limit Errors

```bash
# Check Redis
docker-compose exec redis redis-cli

# Reset tenant rate limits
docker-compose exec app php artisan tinker
app(App\Services\RateLimiterService::class)->reset($tenant);
```

---

## Security Checklist

- [ ] `APP_DEBUG=false` in production
- [ ] Strong `APP_KEY` generated
- [ ] Database credentials secured
- [ ] Redis password set (if public)
- [ ] Firewall configured (allow only 80, 443, 22)
- [ ] SSH key-only authentication
- [ ] Regular security updates (`apt update && apt upgrade`)
- [ ] HTTPS enforced with valid SSL certificate
- [ ] Rate limiting enabled
- [ ] `.env` file not in version control
- [ ] Horizon dashboard requires authentication

---

## Maintenance

### Update Application

```bash
git pull origin main
docker-compose down
docker-compose build --no-cache
docker-compose up -d
docker-compose exec app composer install --no-dev
docker-compose exec app php artisan migrate --force
docker-compose exec app php artisan config:cache
docker-compose exec app php artisan route:cache
docker-compose exec app php artisan view:cache
```

### Clear Caches

```bash
docker-compose exec app php artisan cache:clear
docker-compose exec app php artisan config:clear
docker-compose exec app php artisan route:clear
docker-compose exec app php artisan view:clear
```

---

## Support

For deployment assistance:
- GitHub Issues: https://github.com/yourorg/pxebot/issues
- Email: support@phxbot.com
