<?php

namespace App\Console;

use App\Jobs\FireTimer;
use App\Jobs\PollYouTubeChat;
use App\Jobs\RefreshYouTubeToken;
use App\Models\OauthToken;
use App\Models\Tenant;
use App\Models\Timer;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * Define the application's command schedule.
     */
    protected function schedule(Schedule $schedule): void
    {
        // Poll YouTube chat for running tenants every 2 seconds
        // Note: This requires queue worker concurrency
        $schedule->call(function () {
            $tenants = Tenant::where('status', 'running')->get();
            foreach ($tenants as $tenant) {
                PollYouTubeChat::dispatch($tenant);
            }
        })->everyTwoSeconds()->name('poll-youtube-chats');

        // Fire timers every minute
        $schedule->call(function () {
            $timers = Timer::where('is_enabled', true)
                ->whereHas('tenant', function ($query) {
                    $query->where('status', 'running');
                })
                ->get();

            foreach ($timers as $timer) {
                FireTimer::dispatch($timer);
            }
        })->everyMinute()->name('fire-timers');

        // Refresh expiring tokens every 10 minutes
        $schedule->call(function () {
            $tokens = OauthToken::where('expires_at', '<=', now()->addMinutes(30))
                ->whereNotNull('refresh_token')
                ->get();

            foreach ($tokens as $token) {
                RefreshYouTubeToken::dispatch($token);
            }
        })->everyTenMinutes()->name('refresh-tokens');

        // Cleanup old message logs (keep 7 days)
        $schedule->command('model:prune', ['--model' => 'App\\Models\\MessageLog'])
            ->daily()
            ->name('prune-message-logs');

        // Cleanup old audits (keep 30 days)
        $schedule->command('model:prune', ['--model' => 'App\\Models\\Audit'])
            ->daily()
            ->name('prune-audits');

        // Horizon snapshot
        $schedule->command('horizon:snapshot')->everyFiveMinutes();
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}
