<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Audit;
use App\Models\OauthToken;
use App\Models\Tenant;
use App\Services\YouTubeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Laravel\Socialite\Facades\Socialite;

class GoogleOAuthController extends Controller
{
    public function __construct(
        private YouTubeService $youtube
    ) {}

    /**
     * Redirect to Google OAuth
     */
    public function redirect()
    {
        return Socialite::driver('google')
            ->scopes([
                'https://www.googleapis.com/auth/youtube.readonly',
                'https://www.googleapis.com/auth/youtube.force-ssl',
                'https://www.googleapis.com/auth/youtube',
            ])
            ->with(['access_type' => 'offline', 'prompt' => 'consent'])
            ->redirect();
    }

    /**
     * Handle Google OAuth callback
     */
    public function callback(Request $request)
    {
        try {
            $googleUser = Socialite::driver('google')->user();

            // Get channel info
            $channelInfo = $this->youtube->getChannelInfo($googleUser->token);

            if (!$channelInfo) {
                return redirect()->route('dashboard')
                    ->with('error', 'Failed to fetch YouTube channel information.');
            }

            $user = Auth::user();

            // Create or update tenant (find by channel_id only, update user_id if needed)
            $tenant = Tenant::updateOrCreate(
                [
                    'channel_id' => $channelInfo['channel_id'],
                ],
                [
                    'user_id' => $user->id, // Update user_id to current user
                    'channel_title' => $channelInfo['title'],
                    'channel_handle' => $channelInfo['handle'],
                    'status' => 'stopped',
                ]
            );

            // Store OAuth token
            OauthToken::updateOrCreate(
                [
                    'tenant_id' => $tenant->id,
                    'provider' => 'google',
                ],
                [
                    'access_token' => $googleUser->token,
                    'refresh_token' => $googleUser->refreshToken,
                    'expires_at' => now()->addSeconds($googleUser->expiresIn),
                    'scopes' => [
                        'youtube.readonly',
                        'youtube.force-ssl',
                        'youtube',
                    ],
                ]
            );

            // Log audit
            Audit::log(
                $tenant->id,
                $user->id,
                'youtube_connected',
                ['channel_id' => $channelInfo['channel_id']]
            );

            return redirect()->route('dashboard')
                ->with('success', "YouTube channel '{$channelInfo['title']}' connected successfully!");
        } catch (\Exception $e) {
            return redirect()->route('dashboard')
                ->with('error', 'OAuth connection failed: ' . $e->getMessage());
        }
    }
}
