<?php

namespace App\Http\Controllers;

use App\Services\ActivityHeuristics;
use App\Services\RateLimiterService;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function __construct(
        private ActivityHeuristics $activity,
        private RateLimiterService $rateLimiter
    ) {}

    public function index(Request $request)
    {
        $user = $request->user();
        $tenants = $user->tenants()->with(['oauthTokens'])->get();

        $dashboardData = $tenants->map(function ($tenant) {
            $stats = $this->activity->getStats($tenant);
            $quota = $this->rateLimiter->getRemainingQuota($tenant);

            $recentMessages = $tenant->messagesLog()
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get();

            // Get roasting mode setting from JSON column
            $roastingMode = ($tenant->settings['roasting_mode'] ?? false) === true;

            return [
                'tenant' => $tenant,
                'stats' => $stats,
                'quota' => $quota,
                'recent_messages' => $recentMessages,
                'has_oauth' => $tenant->oauthTokens()->exists(),
                'roasting_mode' => $roastingMode,
            ];
        });

        return view('dashboard.index', [
            'tenants' => $tenants,
            'dashboardData' => $dashboardData,
        ]);
    }
}
