<?php

namespace App\Jobs;

use App\Models\Timer;
use App\Services\ActivityHeuristics;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class FireTimer implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 2;
    public int $timeout = 30;

    public function __construct(
        public Timer $timer
    ) {}

    public function handle(ActivityHeuristics $activity): void
    {
        $tenant = $this->timer->tenant;

        // Skip if tenant not running
        if (!$tenant->isRunning()) {
            return;
        }

        // Check if timer should fire
        if (!$this->timer->shouldFire()) {
            return;
        }

        // Check chat activity
        $activityThreshold = config('services.bot.timer_activity_threshold', 3);
        $activityWindow = $this->timer->chat_activity_window_min;

        if (!$activity->isChatActive($tenant, $activityWindow, $activityThreshold)) {
            Log::debug('Skipping timer - insufficient chat activity', [
                'timer_id' => $this->timer->id,
                'tenant_id' => $tenant->id,
            ]);
            // Still update next fire time
            $this->timer->markFired();
            return;
        }

        // Post timer message
        PostBotReply::dispatch($tenant, $this->timer->message);

        // Mark as fired
        $this->timer->markFired();

        Log::info('Timer fired', [
            'timer_id' => $this->timer->id,
            'tenant_id' => $tenant->id,
            'message' => substr($this->timer->message, 0, 50),
        ]);
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('FireTimer job failed', [
            'timer_id' => $this->timer->id,
            'error' => $exception->getMessage(),
        ]);
    }
}
