<?php

namespace App\Jobs;

use App\Models\MessageLog;
use App\Models\Tenant;
use App\Services\YouTubeService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class PollYouTubeChat implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 30;

    public function __construct(
        public Tenant $tenant
    ) {}

    public function handle(YouTubeService $youtube): void
    {
        // Skip if tenant is not running
        if (!$this->tenant->isRunning()) {
            return;
        }

        // Ensure we have liveChatId
        if (!$this->tenant->live_chat_id) {
            $this->resolveActiveBroadcast($youtube);
            return;
        }

        // Poll messages
        $result = $youtube->pollMessages($this->tenant);

        if (!$result) {
            Log::warning('Failed to poll YouTube messages', ['tenant_id' => $this->tenant->id]);
            return;
        }

        $messages = $result['messages'];
        $newMessagesCount = 0;

        foreach ($messages as $message) {
            $messageId = $message['id'];

            // Skip if already processed
            if ($this->tenant->last_seen_message_id === $messageId) {
                continue;
            }

            // Skip if already in database (idempotency)
            if (MessageLog::where('msg_id', $messageId)->exists()) {
                continue;
            }

            // Dispatch processing job
            ProcessIncomingMessage::dispatch($this->tenant, $message);
            $newMessagesCount++;

            // Update last seen message ID
            $this->tenant->update(['last_seen_message_id' => $messageId]);
        }

        // Update polling timestamp
        $this->tenant->update(['last_polled_at' => now()]);

        if ($newMessagesCount > 0) {
            Log::info('Polled YouTube messages', [
                'tenant_id' => $this->tenant->id,
                'new_messages' => $newMessagesCount,
            ]);
        }
    }

    /**
     * Resolve active broadcast and liveChatId
     */
    private function resolveActiveBroadcast(YouTubeService $youtube): void
    {
        $broadcast = $youtube->getActiveBroadcast($this->tenant);

        if ($broadcast && $broadcast['live_chat_id']) {
            $this->tenant->update([
                'live_chat_id' => $broadcast['live_chat_id'],
                'active_broadcast_id' => $broadcast['broadcast_id'],
                'broadcast_started_at' => $broadcast['started_at'],
            ]);

            Log::info('Active broadcast resolved', [
                'tenant_id' => $this->tenant->id,
                'live_chat_id' => $broadcast['live_chat_id'],
            ]);
        } else {
            Log::warning('No active broadcast found', ['tenant_id' => $this->tenant->id]);
        }
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('PollYouTubeChat job failed', [
            'tenant_id' => $this->tenant->id,
            'error' => $exception->getMessage(),
        ]);

        $this->tenant->update([
            'status' => 'error',
            'last_error' => $exception->getMessage(),
            'last_error_at' => now(),
        ]);
    }
}
