<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Filter extends Model
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'name',
        'type',
        'threshold',
        'action',
        'list',
        'is_enabled',
    ];

    protected function casts(): array
    {
        return [
            'is_enabled' => 'boolean',
            'list' => 'array',
        ];
    }

    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    public function check(string $text): bool
    {
        if (!$this->is_enabled) {
            return false;
        }

        return match ($this->type) {
            'caps' => $this->checkCaps($text),
            'symbols' => $this->checkSymbols($text),
            'links' => $this->checkLinks($text),
            'blacklist' => $this->checkBlacklist($text),
            'emoji' => $this->checkEmoji($text),
            default => false,
        };
    }

    private function checkCaps(string $text): bool
    {
        $upper = preg_match_all('/[A-Z]/', $text);
        $total = strlen(preg_replace('/[^a-zA-Z]/', '', $text));

        if ($total === 0) {
            return false;
        }

        $percentage = ($upper / $total) * 100;
        return $percentage > $this->threshold;
    }

    private function checkSymbols(string $text): bool
    {
        $symbols = preg_match_all('/[^a-zA-Z0-9\s]/', $text);
        $total = strlen($text);

        if ($total === 0) {
            return false;
        }

        $percentage = ($symbols / $total) * 100;
        return $percentage > $this->threshold;
    }

    private function checkLinks(string $text): bool
    {
        $hasLink = preg_match('/https?:\/\/\S+/', $text);
        return (bool) $hasLink;
    }

    private function checkBlacklist(string $text): bool
    {
        if (!$this->list) {
            return false;
        }

        $lowerText = mb_strtolower($text);
        foreach ($this->list as $word) {
            if (str_contains($lowerText, mb_strtolower($word))) {
                return true;
            }
        }

        return false;
    }

    private function checkEmoji(string $text): bool
    {
        $emojiCount = preg_match_all('/[\x{1F600}-\x{1F64F}]/u', $text);
        return $emojiCount > $this->threshold;
    }
}
