<?php

namespace App\Services;

use App\Models\Tenant;
use Illuminate\Support\Facades\Log;
use OpenAI\Laravel\Facades\OpenAI;

class OpenAIService
{
    /**
     * Moderate content using OpenAI moderation endpoint
     */
    public function moderate(string $text): array
    {
        try {
            $response = OpenAI::moderations()->create([
                'input' => $text,
            ]);

            $result = $response->results[0];

            return [
                'flagged' => $result->flagged,
                'categories' => (array) $result->categories,
                'category_scores' => (array) $result->categoryScores,
            ];
        } catch (\Exception $e) {
            Log::error('OpenAI moderation error', ['error' => $e->getMessage()]);
            return [
                'flagged' => false,
                'categories' => [],
                'category_scores' => [],
            ];
        }
    }

    /**
     * Classify message (is it a question? what language?)
     */
    public function classify(string $text): array
    {
        try {
            $response = OpenAI::chat()->create([
                'model' => 'gpt-4o-mini',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a classifier. Analyze the message and detect language INCLUDING script type. ' .
                            'For Urdu written in Roman/Latin script (like "aap kaise ho", "shukriya", "theek hai"), use "ur-Latn". ' .
                            'For Urdu in Arabic script (like "آپ کیسے ہو"), use "ur". ' .
                            'For Hindi in Devanagari use "hi", for Hindi in Roman use "hi-Latn". ' .
                            'Respond ONLY with valid JSON: {"is_question": true/false, "language": "code or code-script", "intent": "brief intent", "script": "roman/arabic/devanagari/latin"}. No other text.',
                    ],
                    [
                        'role' => 'user',
                        'content' => $text,
                    ],
                ],
                'temperature' => 0.3,
                'max_tokens' => 100,
            ]);

            $content = $response->choices[0]->message->content;
            $classification = json_decode($content, true);

            if (!$classification) {
                // Fallback heuristics
                return $this->heuristicClassify($text);
            }

            return [
                'is_question' => $classification['is_question'] ?? false,
                'language' => $classification['language'] ?? 'en',
                'intent' => $classification['intent'] ?? null,
                'script' => $classification['script'] ?? 'latin',
            ];
        } catch (\Exception $e) {
            Log::error('OpenAI classification error', ['error' => $e->getMessage()]);
            return $this->heuristicClassify($text);
        }
    }

    /**
     * Generate response in the same language as the message
     */
    public function respond(Tenant $tenant, string $userMessage, string $detectedLang = 'en', string $script = 'latin', array $classification = []): ?string
    {
        try {
            $channelName = $tenant->channel_title;

            // Check if roasting mode is enabled
            $roastingEnabled = $tenant->settings()
                ->where('key', 'roasting_mode')
                ->value('value') === '1';

            // Build base system prompt
            $systemPrompt = "You are PXEBot, a chatbot assistant for the YouTube channel \"{$channelName}\". ";

            // Language and script instructions
            $systemPrompt .= "You must ALWAYS reply in the same language AND script as the user's message. ";
            $systemPrompt .= "The user's message is in language code: {$detectedLang}. ";

            // Script-specific instructions
            if (str_contains($detectedLang, 'Latn') || $script === 'roman') {
                $systemPrompt .= "The user wrote in ROMAN/LATIN script (not Arabic/Devanagari). Reply using ONLY Roman/Latin script. ";
                $systemPrompt .= "For example, if Urdu: write 'aap kaise hain' NOT 'آپ کیسے ہیں'. ";
            }

            // Roasting mode
            if ($roastingEnabled) {
                $systemPrompt .= "\n\n🔥 ROASTING MODE ACTIVE: ";
                $systemPrompt .= "If the user says something wrong, ignorant, rude, or nonsensical, ROAST them with dark humor and savage wit. ";
                $systemPrompt .= "Be brutally honest, sarcastic, and cutting. Mock their mistakes mercilessly but keep it creative and funny. ";
                $systemPrompt .= "Use clever insults, comparisons, and observations. Think of a comedy roast battle. ";
                $systemPrompt .= "However, if they ask a genuine, respectful question, answer normally. Only roast when deserved. ";
            } else {
                $systemPrompt .= "Be friendly, concise, helpful, and positive. Refuse unsafe requests politely. ";
            }

            $systemPrompt .= "\n\nKeep responses under 200 characters. If you cannot answer, say so briefly in the user's language.";

            $temperature = $roastingEnabled ? 0.9 : (float) config('services.openai.temperature', 0.7);

            $response = OpenAI::chat()->create([
                'model' => config('services.openai.model', 'gpt-4o-mini'),
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => $systemPrompt,
                    ],
                    [
                        'role' => 'user',
                        'content' => $userMessage,
                    ],
                ],
                'temperature' => $temperature,
                'max_tokens' => (int) config('services.openai.max_tokens', 200),
            ]);

            $reply = $response->choices[0]->message->content;

            Log::info('OpenAI response generated', [
                'tenant_id' => $tenant->id,
                'input_length' => strlen($userMessage),
                'output_length' => strlen($reply),
                'language' => $detectedLang,
                'script' => $script,
                'roasting_mode' => $roastingEnabled,
            ]);

            return trim($reply);
        } catch (\Exception $e) {
            Log::error('OpenAI response error', [
                'tenant_id' => $tenant->id,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Heuristic classification fallback
     */
    private function heuristicClassify(string $text): array
    {
        $lowerText = mb_strtolower($text);

        // Detect if it's a question
        $questionMarkers = ['?', 'how', 'what', 'when', 'where', 'who', 'why', 'can', 'is', 'are', 'do', 'does', 'kya', 'kaise', 'kab', 'kahan', 'kyun'];
        $isQuestion = str_contains($text, '?');

        if (!$isQuestion) {
            foreach ($questionMarkers as $marker) {
                if (str_starts_with($lowerText, $marker . ' ')) {
                    $isQuestion = true;
                    break;
                }
            }
        }

        // Simple language detection (very basic)
        $language = 'en';
        $script = 'latin';

        if (preg_match('/[\x{4E00}-\x{9FFF}]/u', $text)) {
            $language = 'zh';
            $script = 'chinese';
        } elseif (preg_match('/[\x{3040}-\x{309F}\x{30A0}-\x{30FF}]/u', $text)) {
            $language = 'ja';
            $script = 'japanese';
        } elseif (preg_match('/[\x{0400}-\x{04FF}]/u', $text)) {
            $language = 'ru';
            $script = 'cyrillic';
        } elseif (preg_match('/[\x{0600}-\x{06FF}]/u', $text)) {
            $language = 'ur';
            $script = 'arabic';
        } elseif (preg_match('/[\x{0900}-\x{097F}]/u', $text)) {
            $language = 'hi';
            $script = 'devanagari';
        } else {
            // Check for Roman Urdu/Hindi patterns
            $romanUrduWords = ['aap', 'kya', 'hai', 'hain', 'kaise', 'theek', 'shukriya', 'bhai', 'yaar', 'bhai', 'kab', 'kahan', 'kyun'];
            foreach ($romanUrduWords as $word) {
                if (str_contains($lowerText, $word)) {
                    $language = 'ur-Latn';
                    $script = 'roman';
                    break;
                }
            }
        }

        return [
            'is_question' => $isQuestion,
            'language' => $language,
            'intent' => $isQuestion ? 'question' : 'statement',
            'script' => $script,
        ];
    }

    /**
     * Check if text should trigger OpenAI response
     */
    public function shouldRespond(string $text, array $classification): bool
    {
        // Always respond to questions
        if ($classification['is_question']) {
            return true;
        }

        // Check if auto-detect is enabled
        if (!config('services.openai.auto_detect_questions', true)) {
            return false;
        }

        // Additional heuristics
        $lowerText = mb_strtolower($text);
        $triggers = ['bot', 'pxebot', 'help', 'explain'];

        foreach ($triggers as $trigger) {
            if (str_contains($lowerText, $trigger)) {
                return true;
            }
        }

        return false;
    }
}
