# 🔥 Roasting Toggle - Debug & Testing Guide

## What Was Fixed ✅

1. ✅ **Database column type** changed from JSON to TEXT
2. ✅ **Model cast** removed
3. ✅ **Settings save** now works (no 500 error)
4. ✅ **Added debug logging** to toggle button
5. ✅ **Added error messages** visible on page
6. ✅ **Created test page** to manually test toggle

---

## 🧪 Testing Instructions

### Option 1: Test on Dashboard (With Debug Logging)

1. **Open Dashboard**: https://phxbot.com/dashboard
2. **Hard Refresh**: Press `Ctrl + Shift + R`
3. **Open Browser Console**: Press `F12` → Go to **Console** tab
4. **Click Toggle Button** (🔥 Roasting Mode)
5. **Watch Console Output**:
   ```
   🔥 Toggle clicked! Current state: false
   URL: https://phxbot.com/tenants/16/settings/toggle/roasting_mode
   CSRF Token: abc123...
   Response status: 200
   Response data: {success: true, enabled: true, message: "..."}
   ✅ Toggle successful! New state: true
   ```

**Expected Behavior**:
- ✅ Console shows all log messages
- ✅ Button changes color (gray → orange or orange → gray)
- ✅ Status text shows "✅ ON" or "❌ OFF"
- ✅ No errors in console
- ✅ No alert popup

**If Toggle Fails**:
- ❌ Error message will show in console
- ❌ Alert popup will appear with error
- ❌ Red error text appears next to button
- 📸 Screenshot console and send to me

---

### Option 2: Test Page (Manual Testing)

1. **Open Test Page**: https://phxbot.com/test-toggle.php
2. **You'll see 3 toggle methods**:

#### Method 1: Alpine.js Toggle (Same as Dashboard)
- Click the toggle switch
- Watch browser console for logs
- If it works here but not on dashboard, Alpine.js might not be loading on dashboard

#### Method 2: Manual PHP Toggle
- Click "Manual Toggle" button
- Page will reload
- Toggle should flip (ON → OFF or OFF → ON)
- Shows database update directly

#### Method 3: Direct Database Update
- Click "Force ON" or "Force OFF"
- Directly sets database value
- Confirms database writes are working

**Test All 3 Methods**:
- ✅ If Method 2 & 3 work → Alpine.js issue
- ✅ If all 3 fail → Database/route issue
- ✅ If all 3 work → Dashboard cache issue

---

## 🔍 Troubleshooting

### Scenario 1: Console Shows "Toggle clicked!" but Nothing Happens

**Problem**: AJAX request isn't being sent

**Check**:
1. Look for **CORS errors** in console
2. Look for **Network errors** in Network tab
3. Check if CSRF token is present

**Fix**: Hard refresh (`Ctrl + Shift + R`)

---

### Scenario 2: Console Shows "HTTP 419" Error

**Problem**: CSRF token mismatch

**Solution**:
```
1. Hard refresh page (Ctrl + Shift + R)
2. If still fails, clear browser cookies
3. Logout and login again
```

---

### Scenario 3: Console Shows "HTTP 403" Error

**Problem**: Authorization failed (you don't own this tenant)

**Solution**:
```bash
# Check tenant ownership
php artisan tinker
```

```php
$tenant = App\Models\Tenant::find(16);
$tenant->user_id; // Should match your user ID
```

---

### Scenario 4: Console Shows "HTTP 500" Error

**Problem**: Server error

**Solution**:
```bash
# Check Laravel logs
tail -50 storage/logs/laravel.log
```

Send me the error from logs.

---

### Scenario 5: Toggle Works but Doesn't Persist

**Problem**: Database update isn't saving

**Test**:
```bash
php artisan tinker
```

```php
$tenant = App\Models\Tenant::find(16);

// Manually set to ON
$tenant->settings()->updateOrCreate(
    ['key' => 'roasting_mode'],
    ['value' => '1']
);

// Check value
$tenant->settings()->where('key', 'roasting_mode')->value('value');
// Should return '1'
```

---

### Scenario 6: Alpine.js Not Working

**Check if Alpine.js is loaded**:
```javascript
// In browser console, type:
window.Alpine
```

**Expected**: Should show Alpine object
**If undefined**: Alpine.js isn't loading

**Fix**:
1. Check `resources/views/layouts/app.blade.php` Line 25
2. Should have: `<script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.x.x/dist/cdn.min.js"></script>`
3. If missing, Alpine.js won't work
4. Hard refresh after checking

---

## 📊 What Each Method Tests

| Method | Tests | Good For |
|--------|-------|----------|
| Dashboard Toggle | Full AJAX + Alpine.js + Routes | Real-world usage |
| Test Page Method 1 | Alpine.js + AJAX isolated | Debugging Alpine |
| Test Page Method 2 | PHP + Database only | Confirming DB works |
| Test Page Method 3 | Direct DB update | Bypassing all logic |

---

## 🎯 Next Steps

### Step 1: Test on Dashboard First

1. Open: https://phxbot.com/dashboard
2. Press `F12` to open console
3. Click toggle
4. **Screenshot console output**
5. Send screenshot to me

### Step 2: If Dashboard Fails, Try Test Page

1. Open: https://phxbot.com/test-toggle.php
2. Try all 3 methods
3. Note which methods work
4. Send results to me

### Step 3: Report Back

**Tell me**:
- ✅ What you see in console when clicking toggle
- ✅ Does button change color?
- ✅ Any error messages?
- ✅ Which test page methods work?
- ✅ Screenshots of console/errors

---

## 📝 Quick Diagnosis Checklist

Run through this checklist and report results:

```
□ Dashboard loads without errors
□ F12 console shows no errors on page load
□ Alpine.js is loaded (type `window.Alpine` in console)
□ Toggle button is visible
□ Clicking toggle shows console log "Toggle clicked!"
□ Console shows URL and CSRF token
□ Console shows "Response status: 200"
□ Console shows response data
□ Button changes color
□ Status text updates (ON/OFF)
□ No alert popup appears
□ Test page Method 2 works (PHP toggle)
□ Test page Method 3 works (Direct DB)
```

**Check all that apply and send me the list!**

---

## 🔧 Manual Database Check

To verify toggle is actually changing database:

```bash
php artisan tinker
```

```php
// Get current value
$tenant = App\Models\Tenant::find(16);
$value = $tenant->settings()->where('key', 'roasting_mode')->value('value');
echo "Current value: " . var_export($value, true);

// Try toggling manually
$newValue = ($value === '1') ? '0' : '1';
$tenant->settings()->updateOrCreate(['key' => 'roasting_mode'], ['value' => $newValue]);

// Confirm it saved
$savedValue = $tenant->settings()->where('key', 'roasting_mode')->value('value');
echo "New value: " . var_export($savedValue, true);
```

If this works but dashboard toggle doesn't → AJAX/Alpine issue
If this fails → Database/permission issue

---

## 🆘 Emergency Quick Fix

If nothing works and you need roasting mode ON **RIGHT NOW**:

```bash
php artisan tinker
```

```php
App\Models\Tenant::find(16)->settings()->updateOrCreate(
    ['key' => 'roasting_mode'],
    ['value' => '1']
);
```

**Done! Roasting mode is now ON** (but we still need to fix the toggle for permanent solution)

---

## 📸 What to Send Me

Please send me:

1. **Screenshot of browser console** when you click toggle
2. **Tell me**: Does button change color? (Yes/No)
3. **Tell me**: Do you see "saving..." text? (Yes/No)
4. **Tell me**: Do you see any alert popup? (Yes/No)
5. **Test page results**: Which methods work? (1, 2, 3, or none)

With this info, I can pinpoint the exact issue!

---

**Summary**: Try dashboard first with F12 console open, screenshot console output, and send to me. If that doesn't reveal the issue, try test-toggle.php page and report which methods work.

**Test Page**: https://phxbot.com/test-toggle.php
**Dashboard**: https://phxbot.com/dashboard

Let's fix this! 🔥
