<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Audit;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Http\Request;

class AdminController extends Controller
{
    // Constructor removed - using route middleware instead

    public function index()
    {
        $stats = [
            'total_tenants' => Tenant::count(),
            'active_tenants' => Tenant::where('status', 'running')->count(),
            'total_commands' => \App\Models\Command::count(),
            'total_messages' => \App\Models\MessageLog::count(),
        ];

        $tenants = Tenant::with('user')
            ->orderBy('updated_at', 'desc')
            ->paginate(20);

        // Get bot account info
        $botAccount = \App\Models\BotAccount::getConnected();

        return view('admin.index', compact('stats', 'tenants', 'botAccount'));
    }

    public function forceStop(Request $request, Tenant $tenant)
    {
        $tenant->update(['status' => 'stopped']);

        Audit::log($tenant->id, $request->user()->id, 'admin_force_stop');

        return back()->with('success', "Bot for '{$tenant->channel_title}' stopped.");
    }

    public function forceStart(Request $request, Tenant $tenant)
    {
        $tenant->update(['status' => 'running']);

        Audit::log($tenant->id, $request->user()->id, 'admin_force_start');

        return back()->with('success', "Bot for '{$tenant->channel_title}' started.");
    }

    public function logs(Tenant $tenant)
    {
        $messages = $tenant->messagesLog()
            ->orderBy('created_at', 'desc')
            ->paginate(50);

        return view('admin.logs', compact('tenant', 'messages'));
    }

    public function audits()
    {
        $audits = Audit::with(['tenant', 'user'])
            ->orderBy('created_at', 'desc')
            ->paginate(50);

        return view('admin.audits', compact('audits'));
    }

    public function quotas()
    {
        $tenants = Tenant::with('user')
            ->orderBy('channel_title')
            ->get();

        return view('admin.quotas', compact('tenants'));
    }

    public function updateQuota(Request $request, Tenant $tenant)
    {
        $request->validate([
            'per_minute' => 'required|integer|min:1|max:1000',
            'per_hour' => 'required|integer|min:1|max:10000',
        ]);

        $quotas = [
            'per_minute' => (int) $request->per_minute,
            'per_hour' => (int) $request->per_hour,
        ];

        $tenant->update(['quotas' => $quotas]);

        Audit::log($tenant->id, $request->user()->id, 'admin_quota_update', [
            'per_minute' => $quotas['per_minute'],
            'per_hour' => $quotas['per_hour'],
        ]);

        return back()->with('success', "Quota updated for '{$tenant->channel_title}'");
    }
}
