<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Audit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;

class SettingsController extends Controller
{
    public function index()
    {
        // Get current API settings from .env
        $settings = [
            'openai_api_key' => config('openai.api_key') ? substr(config('openai.api_key'), 0, 10) . '...' : null,
            'openai_model' => config('services.openai.model', 'gpt-4o-mini'),
            'openai_temperature' => config('services.openai.temperature', 0.7),
            'openai_max_tokens' => config('services.openai.max_tokens', 200),
            'google_client_id' => config('services.google.client_id') ? substr(config('services.google.client_id'), 0, 20) . '...' : null,
            'google_client_secret' => config('services.google.client_secret') ? '••••••••' : null,
        ];

        return view('admin.settings', compact('settings'));
    }

    public function updateOpenAI(Request $request)
    {
        $request->validate([
            'api_key' => 'required|string|min:20',
            'model' => 'required|string|in:gpt-4o-mini,gpt-4o,gpt-4-turbo,gpt-3.5-turbo',
            'temperature' => 'required|numeric|min:0|max:2',
            'max_tokens' => 'required|integer|min:50|max:1000',
        ]);

        try {
            // Update .env file
            $this->updateEnvFile([
                'OPENAI_API_KEY' => $request->api_key,
                'OPENAI_MODEL' => $request->model,
                'OPENAI_TEMPERATURE' => $request->temperature,
                'OPENAI_MAX_TOKENS' => $request->max_tokens,
            ]);

            // Update config/openai.php
            $configPath = config_path('openai.php');
            $configContent = "<?php\n\nreturn [\n    'api_key' => env('OPENAI_API_KEY'),\n    'organization' => env('OPENAI_ORGANIZATION'),\n    'request_timeout' => env('OPENAI_REQUEST_TIMEOUT', 30),\n];\n";
            file_put_contents($configPath, $configContent);

            // Clear config cache
            Artisan::call('config:clear');
            Artisan::call('cache:clear');

            // Log audit
            Audit::log(null, $request->user()->id, 'admin_openai_update', [
                'model' => $request->model,
                'temperature' => $request->temperature,
                'max_tokens' => $request->max_tokens,
            ]);

            return back()->with('success', 'OpenAI settings updated successfully! Config cache cleared.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to update settings: ' . $e->getMessage());
        }
    }

    public function updateGoogle(Request $request)
    {
        $request->validate([
            'client_id' => 'required|string|min:20',
            'client_secret' => 'required|string|min:10',
        ]);

        try {
            // Update .env file
            $this->updateEnvFile([
                'GOOGLE_CLIENT_ID' => $request->client_id,
                'GOOGLE_CLIENT_SECRET' => $request->client_secret,
            ]);

            // Clear config cache
            Artisan::call('config:clear');
            Artisan::call('cache:clear');

            // Log audit
            Audit::log(null, $request->user()->id, 'admin_google_update');

            return back()->with('success', 'Google/YouTube API settings updated successfully! Config cache cleared.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to update settings: ' . $e->getMessage());
        }
    }

    public function testOpenAI(Request $request)
    {
        try {
            $openai = app(\App\Services\OpenAIService::class);

            // Test classification
            $result = $openai->classify('Hello bot, how are you?');

            if ($result && isset($result['language'])) {
                return response()->json([
                    'success' => true,
                    'message' => 'OpenAI API is working!',
                    'test_result' => $result,
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'OpenAI API test failed',
            ], 400);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
            ], 500);
        }
    }

    private function updateEnvFile(array $data)
    {
        $envPath = base_path('.env');

        if (!file_exists($envPath)) {
            throw new \Exception('.env file not found');
        }

        $envContent = file_get_contents($envPath);

        foreach ($data as $key => $value) {
            // Escape special characters in value
            $escapedValue = str_replace('"', '\"', $value);

            // Check if key exists
            if (preg_match("/^{$key}=.*/m", $envContent)) {
                // Update existing key
                $envContent = preg_replace(
                    "/^{$key}=.*/m",
                    "{$key}=\"{$escapedValue}\"",
                    $envContent
                );
            } else {
                // Add new key
                $envContent .= "\n{$key}=\"{$escapedValue}\"";
            }
        }

        file_put_contents($envPath, $envContent);
    }
}
