<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Audit;
use App\Models\BotAccount;
use App\Services\YouTubeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Laravel\Socialite\Facades\Socialite;

class BotAccountOAuthController extends Controller
{
    private YouTubeService $youtube;

    public function __construct(YouTubeService $youtube)
    {
        $this->youtube = $youtube;
    }

    /**
     * Redirect to Google OAuth
     */
    public function redirect()
    {
        // Store in session that this is a bot account OAuth
        session(['oauth_type' => 'bot_account']);

        return Socialite::driver('google')
            ->scopes(['https://www.googleapis.com/auth/youtube.readonly', 'https://www.googleapis.com/auth/youtube.force-ssl', 'https://www.googleapis.com/auth/youtube'])
            ->with(['access_type' => 'offline', 'prompt' => 'consent', 'state' => 'bot_account'])
            ->redirect();
    }

    /**
     * Handle Google OAuth callback
     */
    public function callback(Request $request)
    {
        try {
            // Clear the OAuth type from session
            session()->forget('oauth_type');

            $googleUser = Socialite::driver('google')->user();

            // Get channel info
            $channelInfo = $this->youtube->getChannelInfo($googleUser->token);

            if (!$channelInfo) {
                return redirect()->route('dashboard')
                    ->with('error', 'Failed to fetch YouTube channel information.');
            }

            // Disconnect any existing bot account
            BotAccount::query()->update(['is_connected' => false]);

            // Create or update bot account
            $botAccount = BotAccount::updateOrCreate(
                [
                    'channel_id' => $channelInfo['channel_id'],
                ],
                [
                    'channel_title' => $channelInfo['title'],
                    'channel_handle' => $channelInfo['handle'],
                    'access_token' => $googleUser->token,
                    'refresh_token' => $googleUser->refreshToken,
                    'expires_at' => now()->addSeconds($googleUser->expiresIn),
                    'scopes' => [
                        'youtube.readonly',
                        'youtube.force-ssl',
                        'youtube',
                    ],
                    'is_connected' => true,
                ]
            );

            // Log audit
            Audit::log(
                null,
                Auth::id(),
                'bot_account_connected',
                ['channel_id' => $channelInfo['channel_id'], 'channel_title' => $channelInfo['title']]
            );

            return redirect()->route('admin.index')
                ->with('success', "Bot account '{$channelInfo['title']}' connected successfully! Make sure to add this channel as a moderator in your live chat.");
        } catch (\Exception $e) {
            return redirect()->route('admin.index')
                ->with('error', 'Bot account OAuth connection failed: ' . $e->getMessage());
        }
    }

    /**
     * Disconnect bot account
     */
    public function disconnect()
    {
        $botAccount = BotAccount::getConnected();

        if ($botAccount) {
            $botAccount->update(['is_connected' => false]);

            Audit::log(
                null,
                Auth::id(),
                'bot_account_disconnected',
                ['channel_id' => $botAccount->channel_id]
            );

            return back()->with('success', 'Bot account disconnected successfully!');
        }

        return back()->with('error', 'No bot account connected.');
    }
}
