<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Audit;
use App\Models\OauthToken;
use App\Models\Tenant;
use App\Services\YouTubeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Laravel\Socialite\Facades\Socialite;

class GoogleOAuthController extends Controller
{
    public function __construct(
        private YouTubeService $youtube
    ) {}

    /**
     * Redirect to Google OAuth
     */
    public function redirect()
    {
        return Socialite::driver('google')
            ->scopes([
                'https://www.googleapis.com/auth/youtube.readonly',
                'https://www.googleapis.com/auth/youtube.force-ssl',
                'https://www.googleapis.com/auth/youtube',
            ])
            ->with(['access_type' => 'offline', 'prompt' => 'consent'])
            ->redirect();
    }

    /**
     * Handle Google OAuth callback
     */
    public function callback(Request $request)
    {
        try {
            // Check if this is a bot account OAuth flow
            if (session('oauth_type') === 'bot_account') {
                // Handle bot account connection
                return $this->handleBotAccountCallback($request);
            }

            $googleUser = Socialite::driver('google')->user();

            // Get channel info
            $channelInfo = $this->youtube->getChannelInfo($googleUser->token);

            if (!$channelInfo) {
                return redirect()->route('login')
                    ->with('error', 'Failed to fetch YouTube channel information.');
            }

            // Check if this is the bot account email (vanihbx@gmail.com)
            if ($googleUser->email === 'vanihbx@gmail.com') {
                return $this->handleBotAccountConnection($googleUser, $channelInfo);
            }

            // Check if this is the bot account channel - prevent creating tenant for bot
            $botAccount = \App\Models\BotAccount::where('channel_id', $channelInfo['channel_id'])->first();
            if ($botAccount) {
                return redirect()->route('login')
                    ->with('error', 'This is the bot account channel. Please login as admin to manage the bot.');
            }

            // Find or create user based on Google email
            $user = \App\Models\User::firstOrCreate(
                ['email' => $googleUser->email],
                [
                    'name' => $googleUser->name,
                    'password' => \Illuminate\Support\Facades\Hash::make(\Illuminate\Support\Str::random(32)),
                    'is_admin' => false,
                ]
            );

            // Log the user in with permanent session
            Auth::login($user, true);

            // Create or update tenant (find by channel_id only, update user_id if needed)
            $tenant = Tenant::updateOrCreate(
                [
                    'channel_id' => $channelInfo['channel_id'],
                ],
                [
                    'user_id' => $user->id, // Update user_id to current user
                    'channel_title' => $channelInfo['title'],
                    'channel_handle' => $channelInfo['handle'],
                    'status' => 'stopped',
                ]
            );

            // Store OAuth token
            OauthToken::updateOrCreate(
                [
                    'tenant_id' => $tenant->id,
                    'provider' => 'google',
                ],
                [
                    'access_token' => $googleUser->token,
                    'refresh_token' => $googleUser->refreshToken,
                    'expires_at' => now()->addSeconds($googleUser->expiresIn),
                    'scopes' => [
                        'youtube.readonly',
                        'youtube.force-ssl',
                        'youtube',
                    ],
                ]
            );

            // Log audit
            Audit::log(
                $tenant->id,
                $user->id,
                'youtube_connected',
                ['channel_id' => $channelInfo['channel_id']]
            );

            return redirect()->route('dashboard')
                ->with('success', "YouTube channel '{$channelInfo['title']}' connected successfully!");
        } catch (\Exception $e) {
            return redirect()->route('login')
                ->with('error', 'OAuth connection failed: ' . $e->getMessage());
        }
    }

    /**
     * Handle bot account OAuth callback
     */
    private function handleBotAccountCallback(Request $request)
    {
        try {
            // Clear the OAuth type from session
            session()->forget('oauth_type');

            $googleUser = Socialite::driver('google')->user();

            \Log::info('Bot Account OAuth - Google User:', [
                'email' => $googleUser->email,
                'name' => $googleUser->name,
                'token_exists' => !empty($googleUser->token),
                'refresh_token_exists' => !empty($googleUser->refreshToken),
            ]);

            // Get channel info
            $channelInfo = $this->youtube->getChannelInfo($googleUser->token);

            \Log::info('Bot Account OAuth - Channel Info:', ['channel_info' => $channelInfo]);

            if (!$channelInfo) {
                return redirect()->route('admin.index')
                    ->with('error', 'Failed to fetch YouTube channel information.');
            }

            // Disconnect any existing bot account
            \App\Models\BotAccount::query()->update(['is_connected' => false]);

            // Create or update bot account
            $botAccount = \App\Models\BotAccount::updateOrCreate(
                [
                    'channel_id' => $channelInfo['channel_id'],
                ],
                [
                    'channel_title' => $channelInfo['title'],
                    'channel_handle' => $channelInfo['handle'],
                    'access_token' => $googleUser->token,
                    'refresh_token' => $googleUser->refreshToken,
                    'expires_at' => now()->addSeconds($googleUser->expiresIn),
                    'scopes' => [
                        'youtube.readonly',
                        'youtube.force-ssl',
                        'youtube',
                    ],
                    'is_connected' => true,
                ]
            );

            // Log audit
            Audit::log(
                null,
                Auth::id(),
                'bot_account_connected',
                ['channel_id' => $channelInfo['channel_id'], 'channel_title' => $channelInfo['title']]
            );

            return redirect()->route('admin.index')
                ->with('success', "Bot account '{$channelInfo['title']}' connected successfully! Make sure to add this channel as a moderator in your live chat.");
        } catch (\Exception $e) {
            \Log::error('Bot Account OAuth Error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()->route('admin.index')
                ->with('error', 'Bot account OAuth connection failed: ' . $e->getMessage() . ' (Check logs for details)');
        }
    }

    /**
     * Handle bot account connection from regular OAuth flow (vanihbx@gmail.com)
     */
    private function handleBotAccountConnection($googleUser, $channelInfo)
    {
        try {
            \Log::info('Bot Account Auto-Connection:', [
                'email' => $googleUser->email,
                'channel' => $channelInfo['title'],
            ]);

            // Disconnect any existing bot account
            \App\Models\BotAccount::query()->update(['is_connected' => false]);

            // Create or update bot account in bot_account table
            $botAccount = \App\Models\BotAccount::updateOrCreate(
                [
                    'channel_id' => $channelInfo['channel_id'],
                ],
                [
                    'channel_title' => $channelInfo['title'],
                    'channel_handle' => $channelInfo['handle'],
                    'access_token' => $googleUser->token,
                    'refresh_token' => $googleUser->refreshToken,
                    'expires_at' => now()->addSeconds($googleUser->expiresIn),
                    'scopes' => [
                        'youtube.readonly',
                        'youtube.force-ssl',
                        'youtube',
                    ],
                    'is_connected' => true,
                ]
            );

            // Check if admin user exists, otherwise login as regular user for first time
            $adminUser = \App\Models\User::where('email', 'admin@daybot.tv')->first();

            if ($adminUser) {
                // Admin exists, login as admin
                Auth::login($adminUser, true);

                // Log audit
                Audit::log(
                    null,
                    $adminUser->id,
                    'bot_account_connected',
                    ['channel_id' => $channelInfo['channel_id'], 'channel_title' => $channelInfo['title']]
                );

                return redirect()->route('admin.index')
                    ->with('success', "Bot account '{$channelInfo['title']}' connected successfully! You are now logged in as admin.");
            } else {
                // Admin doesn't exist yet, show message
                return redirect()->route('login')
                    ->with('success', "Bot account '{$channelInfo['title']}' connected successfully! Please login with admin@daybot.tv to manage the bot.");
            }
        } catch (\Exception $e) {
            \Log::error('Bot Account Auto-Connection Error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()->route('login')
                ->with('error', 'Bot account connection failed: ' . $e->getMessage());
        }
    }
}
