<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Models\BannedKeyword;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class BannedKeywordController extends Controller
{
    public function index(Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $keywords = $tenant->bannedKeywords()->orderBy('created_at', 'desc')->get();
        $warnings = $tenant->userWarnings()
            ->where('violation_type', 'banned_keyword')
            ->where('expires_at', '>', now())
            ->orderBy('last_violation_at', 'desc')
            ->get();

        return view('tenant.banned-keywords.index', compact('tenant', 'keywords', 'warnings'));
    }

    public function store(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $validated = $request->validate([
            'keyword' => 'required|string|max:255',
            'timeout_duration' => 'required|integer|min:60|max:86400', // 1 minute to 24 hours
        ]);

        $tenant->bannedKeywords()->create([
            'keyword' => $validated['keyword'],
            'timeout_duration' => $validated['timeout_duration'],
            'is_enabled' => true,
        ]);

        return back()->with('success', 'Banned keyword added successfully!');
    }

    public function update(Request $request, Tenant $tenant, BannedKeyword $bannedKeyword)
    {
        Gate::authorize('manage', $tenant);

        if ($bannedKeyword->tenant_id !== $tenant->id) {
            abort(403);
        }

        $validated = $request->validate([
            'keyword' => 'required|string|max:255',
            'timeout_duration' => 'required|integer|min:60|max:86400',
            'is_enabled' => 'boolean',
        ]);

        $bannedKeyword->update($validated);

        return back()->with('success', 'Banned keyword updated successfully!');
    }

    public function destroy(Tenant $tenant, BannedKeyword $bannedKeyword)
    {
        Gate::authorize('manage', $tenant);

        if ($bannedKeyword->tenant_id !== $tenant->id) {
            abort(403);
        }

        $bannedKeyword->delete();

        return back()->with('success', 'Banned keyword deleted successfully!');
    }

    public function toggle(Tenant $tenant, BannedKeyword $bannedKeyword)
    {
        Gate::authorize('manage', $tenant);

        if ($bannedKeyword->tenant_id !== $tenant->id) {
            abort(403);
        }

        $bannedKeyword->update([
            'is_enabled' => !$bannedKeyword->is_enabled,
        ]);

        return back()->with('success', 'Banned keyword ' . ($bannedKeyword->is_enabled ? 'enabled' : 'disabled') . '!');
    }
}
