<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Models\Audit;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class TenantController extends Controller
{
    /**
     * Start bot for tenant
     */
    public function start(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        if (!$tenant->oauthTokens()->exists()) {
            return back()->with('error', 'Please connect your YouTube channel first.');
        }

        $tenant->update([
            'status' => 'running',
            'last_error' => null,
            'last_error_at' => null,
            'last_seen_message_id' => null, // Reset to avoid processing old messages
            'live_chat_id' => null, // Force fetch new live chat ID
        ]);

        Audit::log($tenant->id, $request->user()->id, 'bot_started');

        // Send welcome message to stream chat
        dispatch(function () use ($tenant) {
            sleep(2); // Wait 2 seconds for bot to fully initialize

            $youtubeService = app(\App\Services\YouTubeService::class);

            // Get live chat ID if not set
            if (!$tenant->live_chat_id) {
                $broadcast = $youtubeService->getActiveBroadcast($tenant);
                if ($broadcast && $broadcast['live_chat_id']) {
                    $tenant->update(['live_chat_id' => $broadcast['live_chat_id']]);
                }
            }

            // Send welcome message
            if ($tenant->live_chat_id) {
                $welcomeMessage = "🤖 DAYBot is now connected and ready! Type 'bot hello' to test me.";
                $youtubeService->sendMessage($tenant, $welcomeMessage);
            }
        })->afterResponse();

        return back()->with('success', 'Bot started successfully!');
    }

    /**
     * Stop bot for tenant
     */
    public function stop(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $tenant->update(['status' => 'stopped']);

        Audit::log($tenant->id, $request->user()->id, 'bot_stopped');

        return back()->with('success', 'Bot stopped successfully.');
    }

    /**
     * Delete tenant
     */
    public function destroy(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $tenant->update(['status' => 'stopped']);
        $tenant->delete();

        Audit::log(null, $request->user()->id, 'tenant_deleted', [
            'tenant_id' => $tenant->id,
            'channel_id' => $tenant->channel_id,
        ]);

        return redirect()->route('dashboard')
            ->with('success', 'Channel disconnected successfully.');
    }

    /**
     * Toggle a boolean setting
     */
    public function toggleSetting(Request $request, Tenant $tenant, string $key)
    {
        Gate::authorize('manage', $tenant);

        // Validate setting key
        $allowedSettings = ['roasting_mode'];

        if (!in_array($key, $allowedSettings)) {
            return response()->json(['error' => 'Invalid setting key'], 400);
        }

        // Get current value
        $setting = $tenant->settings()->firstOrNew(['key' => $key]);
        $currentValue = $setting->value ?? '0';

        // Toggle value
        $newValue = $currentValue === '1' ? '0' : '1';

        $tenant->settings()->updateOrCreate(
            ['key' => $key],
            ['value' => $newValue]
        );

        // Log the action
        Audit::log($tenant->id, $request->user()->id, 'setting_toggled', [
            'setting' => $key,
            'new_value' => $newValue,
        ]);

        return response()->json([
            'success' => true,
            'enabled' => $newValue === '1',
            'message' => ucfirst(str_replace('_', ' ', $key)) . ' ' . ($newValue === '1' ? 'enabled' : 'disabled'),
        ]);
    }

    /**
     * Update bot settings (roasting level, response length)
     */
    public function updateSettings(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        // Hardcoded settings - no user control
        $tenant->settings()->updateOrCreate(
            ['key' => 'roasting_level'],
            ['value' => 'dark']
        );

        $tenant->settings()->updateOrCreate(
            ['key' => 'response_length'],
            ['value' => 'medium']
        );

        // Log the action
        Audit::log($tenant->id, $request->user()->id, 'bot_settings_updated', [
            'roasting_level' => 'dark',
            'response_length' => 'medium',
        ]);

        return back()->with('success', 'Bot settings updated successfully!');
    }

    /**
     * Update stream and personal info
     */
    public function updateStreamInfo(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $request->validate([
            'stream_type' => 'nullable|in:gaming,tech,music,cooking,fitness,vlog,education,comedy,art,sports,other',
            'current_game' => 'nullable|string|max:255',
            'personal_info' => 'nullable|string|max:2000',
        ]);

        $description = null;
        if ($request->stream_type) {
            $description = match($request->stream_type) {
                'gaming' => $request->current_game ? "Gaming channel playing {$request->current_game}" : "Gaming channel",
                'tech' => "Technology and tech content",
                'music' => "Music and entertainment",
                'cooking' => "Cooking and food content",
                'fitness' => "Fitness and workout content",
                'vlog' => "Vlog and lifestyle content",
                'education' => "Educational content",
                'comedy' => "Comedy and entertainment",
                'art' => "Art and creative content",
                'sports' => "Sports content",
                default => "General content",
            };
        }

        $tenant->update([
            'stream_type' => $request->stream_type,
            'current_game' => $request->current_game,
            'stream_description' => $description,
            'personal_info' => $request->personal_info,
            'stream_context_updated_at' => now(),
        ]);

        // Log the action
        Audit::log($tenant->id, $request->user()->id, 'stream_info_updated', [
            'stream_type' => $request->stream_type,
            'current_game' => $request->current_game,
            'has_personal_info' => !empty($request->personal_info),
        ]);

        return back()->with('success', 'Stream & personal info updated successfully!');
    }

    /**
     * Update spam protection settings
     */
    public function updateSpamProtection(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $validated = $request->validate([
            'spam_protection_enabled' => 'required|boolean',
            'spam_message_limit' => 'required|integer|min:2|max:10',
            'spam_time_window' => 'required|integer|min:10|max:120',
            'spam_timeout_duration' => 'required|integer|min:60|max:86400',
        ]);

        $tenant->update($validated);

        Audit::log($tenant->id, $request->user()->id, 'spam_protection_updated', $validated);

        return back()->with('success', 'Spam protection settings updated successfully!');
    }
}
