<?php

namespace App\Jobs;

use App\Models\MessageLog;
use App\Models\Tenant;
use App\Services\StreamContextService;
use App\Services\YouTubeService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class PollYouTubeChat implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 30;

    public function __construct(
        public Tenant $tenant
    ) {}

    public function handle(YouTubeService $youtube, StreamContextService $streamContext): void
    {
        // Skip if tenant is not running
        if (!$this->tenant->isRunning()) {
            return;
        }

        // Ensure we have liveChatId - keep trying to find active stream
        if (!$this->tenant->live_chat_id) {
            $this->resolveActiveBroadcast($youtube);

            // If still no live chat ID, keep status as running (waiting for stream)
            if (!$this->tenant->fresh()->live_chat_id) {
                $this->tenant->update([
                    'last_error' => 'Waiting for live stream to start...',
                    'last_error_at' => now(),
                ]);

                Log::info('Bot waiting for stream', ['tenant_id' => $this->tenant->id]);
            }
            return;
        }

        // Detect/update stream context if needed (every 30 minutes)
        if ($streamContext->shouldRefreshContext($this->tenant)) {
            $streamContext->detectAndUpdateContext($this->tenant);
        }

        // Poll messages
        $result = $youtube->pollMessages($this->tenant);

        if (!$result) {
            // Polling failed - likely stream ended, clear live_chat_id to find new stream
            Log::info('Stream ended, waiting for next stream', ['tenant_id' => $this->tenant->id]);

            $this->tenant->update([
                'live_chat_id' => null,
                'active_broadcast_id' => null,
                'last_seen_message_id' => null, // Reset for next stream
                'last_error' => 'Stream ended. Waiting for next live stream...',
                'last_error_at' => now(),
            ]);
            return;
        }

        // Clear error if polling successful
        if ($this->tenant->last_error) {
            $this->tenant->update([
                'last_error' => null,
                'last_error_at' => null,
            ]);
        }

        $messages = $result['messages'];
        $newMessagesCount = 0;

        // Get bot account channel ID to ignore bot's own messages
        $botAccount = \App\Models\BotAccount::getConnected();
        $botChannelId = $botAccount ? $botAccount->channel_id : null;

        foreach ($messages as $message) {
            $messageId = $message['id'];
            $authorChannelId = $message['authorDetails']['channelId'] ?? null;

            // CRITICAL: Ignore bot's own messages at polling level
            if ($authorChannelId === $botChannelId || $authorChannelId === $this->tenant->channel_id) {
                continue;
            }

            // Skip if already in database (idempotency) - PRIMARY duplicate check
            if (MessageLog::where('msg_id', $messageId)->exists()) {
                continue;
            }

            // Additional check: Skip messages older than 2 minutes to avoid processing old messages
            $publishedAt = $message['snippet']['publishedAt'] ?? null;
            if ($publishedAt) {
                $messageTime = \Carbon\Carbon::parse($publishedAt);
                $minutesOld = now()->diffInMinutes($messageTime);

                // Only process messages from last 2 minutes
                if ($minutesOld > 2) {
                    Log::info('Skipping old message', [
                        'tenant_id' => $this->tenant->id,
                        'message_id' => $messageId,
                        'minutes_old' => $minutesOld,
                    ]);
                    continue;
                }
            }

            // Dispatch processing job
            ProcessIncomingMessage::dispatch($this->tenant, $message);
            $newMessagesCount++;

            // Update last seen message ID
            $this->tenant->update(['last_seen_message_id' => $messageId]);
        }

        // Update polling timestamp
        $this->tenant->update(['last_polled_at' => now()]);

        if ($newMessagesCount > 0) {
            Log::info('Polled YouTube messages', [
                'tenant_id' => $this->tenant->id,
                'new_messages' => $newMessagesCount,
            ]);
        }
    }

    /**
     * Resolve active broadcast and liveChatId
     */
    private function resolveActiveBroadcast(YouTubeService $youtube): void
    {
        $broadcast = $youtube->getActiveBroadcast($this->tenant);

        if ($broadcast && $broadcast['live_chat_id']) {
            $this->tenant->update([
                'live_chat_id' => $broadcast['live_chat_id'],
                'active_broadcast_id' => $broadcast['broadcast_id'],
                'broadcast_started_at' => $broadcast['started_at'],
            ]);

            Log::info('Active broadcast resolved', [
                'tenant_id' => $this->tenant->id,
                'live_chat_id' => $broadcast['live_chat_id'],
            ]);
        } else {
            Log::warning('No active broadcast found', ['tenant_id' => $this->tenant->id]);
        }
    }

    public function failed(\Throwable $exception): void
    {
        $errorMessage = $exception->getMessage();

        Log::error('PollYouTubeChat job failed', [
            'tenant_id' => $this->tenant->id,
            'error' => $errorMessage,
        ]);

        // Check if error is recoverable (stream ended, no broadcast)
        $recoverableErrors = [
            'live chat is no longer live',
            'liveChatEnded',
            'No active broadcast',
        ];

        $isRecoverable = false;
        foreach ($recoverableErrors as $pattern) {
            if (stripos($errorMessage, $pattern) !== false) {
                $isRecoverable = true;
                break;
            }
        }

        if ($isRecoverable) {
            // Keep bot running, just waiting for next stream
            $this->tenant->update([
                'live_chat_id' => null,
                'active_broadcast_id' => null,
                'last_seen_message_id' => null,
                'last_error' => 'Stream ended. Waiting for next live stream...',
                'last_error_at' => now(),
            ]);

            Log::info('Recoverable error, bot will auto-reconnect', [
                'tenant_id' => $this->tenant->id,
            ]);
        } else {
            // Actual error that needs user attention (OAuth, permissions, etc)
            $this->tenant->update([
                'status' => 'error',
                'last_error' => $errorMessage,
                'last_error_at' => now(),
            ]);
        }
    }
}
