<?php

namespace App\Services;

use App\Models\Tenant;
use Illuminate\Support\Facades\DB;

class ActivityHeuristics
{
    /**
     * Check if chat has been active recently
     */
    public function isChatActive(Tenant $tenant, int $windowMinutes = 10, int $threshold = 3): bool
    {
        $since = now()->subMinutes($windowMinutes);

        $messageCount = DB::table('messages_log')
            ->where('tenant_id', $tenant->id)
            ->where('created_at', '>=', $since)
            ->count();

        return $messageCount >= $threshold;
    }

    /**
     * Get message count in time window
     */
    public function getMessageCount(Tenant $tenant, int $windowMinutes): int
    {
        $since = now()->subMinutes($windowMinutes);

        return DB::table('messages_log')
            ->where('tenant_id', $tenant->id)
            ->where('created_at', '>=', $since)
            ->count();
    }

    /**
     * Get recent message authors (unique viewers)
     */
    public function getRecentViewers(Tenant $tenant, int $windowMinutes = 60): int
    {
        $since = now()->subMinutes($windowMinutes);

        return DB::table('messages_log')
            ->where('tenant_id', $tenant->id)
            ->where('created_at', '>=', $since)
            ->distinct('author_channel_id')
            ->count('author_channel_id');
    }

    /**
     * Get activity stats
     */
    public function getStats(Tenant $tenant): array
    {
        return [
            'last_1min' => $this->getMessageCount($tenant, 1),
            'last_5min' => $this->getMessageCount($tenant, 5),
            'last_10min' => $this->getMessageCount($tenant, 10),
            'last_hour' => $this->getMessageCount($tenant, 60),
            'unique_viewers_1h' => $this->getRecentViewers($tenant, 60),
        ];
    }
}
