<?php

namespace App\Services;

use App\Models\Command;
use App\Models\Tenant;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class CommandRuntime
{
    private array $builtInCommands = [
        'help',
        'commands',
        'uptime',
        'discord',
        'lurk',
        'unlurk',
        'so',
        'rules',
        'ping',
    ];

    public function __construct(
        private ActivityHeuristics $activityHeuristics
    ) {}

    /**
     * Parse and execute command if found
     */
    public function execute(Tenant $tenant, string $text, array $context = []): ?string
    {
        if (!str_starts_with($text, '!')) {
            return null;
        }

        $parts = explode(' ', trim($text), 2);
        $commandName = strtolower(ltrim($parts[0], '!'));
        $query = $parts[1] ?? '';

        // Check built-in commands first
        if (in_array($commandName, $this->builtInCommands)) {
            return $this->executeBuiltIn($tenant, $commandName, $query, $context);
        }

        // Check custom commands
        $command = $this->findCustomCommand($tenant, $commandName);
        if ($command) {
            return $this->executeCustom($command, $query, $context);
        }

        return null;
    }

    /**
     * Find custom command by name or alias
     */
    private function findCustomCommand(Tenant $tenant, string $name): ?Command
    {
        // Try exact name match
        $command = $tenant->commands()
            ->where('is_enabled', true)
            ->where('name', $name)
            ->first();

        if ($command) {
            return $command;
        }

        // Try alias match
        $commands = $tenant->commands()
            ->where('is_enabled', true)
            ->whereNotNull('aliases')
            ->get();

        foreach ($commands as $cmd) {
            if ($cmd->aliases && in_array($name, $cmd->aliases)) {
                return $cmd;
            }
        }

        return null;
    }

    /**
     * Execute custom command
     */
    private function executeCustom(Command $command, string $query, array $context): ?string
    {
        // Check cooldown (per-command)
        if ($command->isOnCooldown()) {
            return null;
        }

        // Check user cooldown (global anti-spam)
        $userCooldownKey = "user_cooldown:{$context['tenant_id']}:{$context['author_id']}";
        if (Cache::has($userCooldownKey)) {
            return null;
        }

        // Expand variables
        $response = $this->expandVariables($command->response, $query, $context, $command->variables);

        // Record usage
        $command->incrementUseCount();

        // Set user cooldown
        $cooldownSec = config('services.bot.user_cooldown', 10);
        Cache::put($userCooldownKey, true, $cooldownSec);

        return $response;
    }

    /**
     * Execute built-in command
     */
    private function executeBuiltIn(Tenant $tenant, string $command, string $query, array $context): ?string
    {
        return match ($command) {
            'help' => $this->handleHelp($tenant),
            'commands' => $this->handleCommandsList($tenant),
            'uptime' => $this->handleUptime($tenant),
            'discord' => $this->handleDiscord($tenant),
            'lurk' => $this->handleLurk($context),
            'unlurk' => $this->handleUnlurk($context),
            'so' => $this->handleShoutout($query, $context),
            'rules' => $this->handleRules($tenant),
            'ping' => $this->handlePing(),
            default => null,
        };
    }

    /**
     * Expand variables in command response
     */
    private function expandVariables(string $template, string $query, array $context, ?array $customVars = null): string
    {
        $variables = [
            '{user}' => $context['author_name'] ?? 'User',
            '{channel}' => $context['channel_name'] ?? 'Channel',
            '{query}' => $query,
            '{uptime}' => $this->formatUptime($context['uptime'] ?? 0),
            '{count}' => $context['use_count'] ?? 0,
            '{random}' => rand(1, 100),
            '{subscribers}' => $context['subscribers'] ?? 0,
            '{viewers}' => $context['viewers'] ?? 0,
        ];

        // Add custom variables
        if ($customVars) {
            foreach ($customVars as $key => $value) {
                $variables["{{$key}}"] = $value;
            }
        }

        return str_replace(array_keys($variables), array_values($variables), $template);
    }

    /**
     * Built-in command handlers
     */
    private function handleHelp(Tenant $tenant): string
    {
        return "DAYBot commands: !help, !commands, !uptime, !discord, !lurk, !so <user>, !rules, !ping";
    }

    private function handleCommandsList(Tenant $tenant): string
    {
        $commands = $tenant->commands()
            ->where('is_enabled', true)
            ->limit(10)
            ->pluck('name')
            ->toArray();

        if (empty($commands)) {
            return "No custom commands available yet.";
        }

        return "Available commands: !" . implode(', !', $commands);
    }

    private function handleUptime(Tenant $tenant): string
    {
        $uptime = $tenant->getUptime();
        if (!$uptime) {
            return "Stream is not currently live.";
        }

        return "Stream uptime: " . $this->formatUptime($uptime);
    }

    private function handleDiscord(Tenant $tenant): string
    {
        $discordUrl = $tenant->getSetting('discord_url');
        if (!$discordUrl) {
            return "Discord link not configured yet.";
        }

        return "Join our Discord: " . $discordUrl;
    }

    private function handleLurk(array $context): string
    {
        return "Thanks for lurking, {$context['author_name']}! 👀";
    }

    private function handleUnlurk(array $context): string
    {
        return "Welcome back, {$context['author_name']}! 🎉";
    }

    private function handleShoutout(string $query, array $context): string
    {
        $targetUser = trim($query);
        if (empty($targetUser)) {
            return "Usage: !so <username>";
        }

        return "Shoutout to {$targetUser}! Go check out their channel! 🌟";
    }

    private function handleRules(Tenant $tenant): string
    {
        $rules = $tenant->getSetting('rules_text');
        if (!$rules) {
            return "Channel rules: Be respectful, have fun, and follow YouTube's community guidelines!";
        }

        return (string) $rules;
    }

    private function handlePing(): string
    {
        return "Pong! 🏓 DAYBot is online and ready.";
    }

    /**
     * Format uptime in human-readable format
     */
    private function formatUptime(int $seconds): string
    {
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $secs = $seconds % 60;

        if ($hours > 0) {
            return sprintf('%dh %dm', $hours, $minutes);
        }

        if ($minutes > 0) {
            return sprintf('%dm %ds', $minutes, $secs);
        }

        return sprintf('%ds', $secs);
    }
}
