<?php

namespace App\Services;

use App\Models\BannedKeyword;
use App\Models\MessageLog;
use App\Models\Tenant;
use App\Models\UserWarning;
use Illuminate\Support\Facades\Log;

class ModerationService
{
    public function __construct(
        private YouTubeService $youtubeService
    ) {}

    /**
     * Check banned keywords and handle violations
     * Returns response message if keyword detected, null otherwise
     */
    public function checkBannedKeywords(Tenant $tenant, string $text, string $userChannelId, string $userName): ?array
    {
        $bannedKeywords = $tenant->bannedKeywords()->where('is_enabled', true)->get();

        foreach ($bannedKeywords as $keyword) {
            if ($keyword->matches($text)) {
                // Get or create warning record
                $warning = UserWarning::firstOrNew([
                    'tenant_id' => $tenant->id,
                    'user_channel_id' => $userChannelId,
                    'violation_type' => 'banned_keyword',
                ]);

                // Check if this is first offense or second+
                if (!$warning->exists || !$warning->isActive()) {
                    // First warning
                    $warning->fill([
                        'user_name' => $userName,
                        'warning_count' => 1,
                        'last_violation_at' => now(),
                        'violation_details' => "Used banned keyword: {$keyword->keyword}",
                        'expires_at' => now()->addHours(24),
                    ])->save();

                    Log::info('User warned for banned keyword', [
                        'tenant_id' => $tenant->id,
                        'user' => $userName,
                        'keyword' => $keyword->keyword,
                    ]);

                    return [
                        'action' => 'warn',
                        'message' => "⚠️ @{$userName} Please avoid using inappropriate language. This is your first warning.",
                    ];
                } else {
                    // Second+ offense - timeout user
                    $warning->incrementWarning("Used banned keyword again: {$keyword->keyword}");

                    $timeoutSuccess = $this->youtubeService->banUser(
                        $tenant,
                        $userChannelId,
                        $keyword->timeout_duration
                    );

                    if ($timeoutSuccess) {
                        Log::info('User timed out for repeated banned keyword violation', [
                            'tenant_id' => $tenant->id,
                            'user' => $userName,
                            'keyword' => $keyword->keyword,
                            'duration' => $keyword->timeout_duration,
                        ]);

                        $minutes = round($keyword->timeout_duration / 60);
                        return [
                            'action' => 'timeout',
                            'message' => "🚫 @{$userName} has been timed out for {$minutes} minutes for repeated violations.",
                        ];
                    } else {
                        Log::error('Failed to timeout user', [
                            'tenant_id' => $tenant->id,
                            'user' => $userName,
                        ]);
                        return null;
                    }
                }
            }
        }

        return null;
    }

    /**
     * Check for spam (repeated messages)
     * Returns response message if spam detected, null otherwise
     */
    public function checkSpam(Tenant $tenant, string $text, string $userChannelId, string $userName): ?array
    {
        if (!$tenant->spam_protection_enabled) {
            return null;
        }

        $limit = $tenant->spam_message_limit ?? 3;
        $window = $tenant->spam_time_window ?? 30;

        // Count same messages from this user in time window
        $recentSameMessages = MessageLog::where('tenant_id', $tenant->id)
            ->where('author_channel_id', $userChannelId)
            ->where('text', $text)
            ->where('created_at', '>=', now()->subSeconds($window))
            ->count();

        if ($recentSameMessages >= $limit) {
            // Get or create spam warning
            $warning = UserWarning::firstOrNew([
                'tenant_id' => $tenant->id,
                'user_channel_id' => $userChannelId,
                'violation_type' => 'spam',
            ]);

            if (!$warning->exists || !$warning->isActive()) {
                // First spam warning
                $warning->fill([
                    'user_name' => $userName,
                    'warning_count' => 1,
                    'last_violation_at' => now(),
                    'violation_details' => "Spammed message: " . substr($text, 0, 50),
                    'expires_at' => now()->addHours(24),
                ])->save();

                Log::info('User warned for spam', [
                    'tenant_id' => $tenant->id,
                    'user' => $userName,
                    'message_count' => $recentSameMessages,
                ]);

                return [
                    'action' => 'warn',
                    'message' => "⚠️ @{$userName} Please don't spam the chat. This is your first warning.",
                ];
            } else {
                // Second+ offense - timeout
                $warning->incrementWarning("Repeated spam: " . substr($text, 0, 50));

                $duration = $tenant->spam_timeout_duration ?? 300;
                $timeoutSuccess = $this->youtubeService->banUser(
                    $tenant,
                    $userChannelId,
                    $duration
                );

                if ($timeoutSuccess) {
                    Log::info('User timed out for spam', [
                        'tenant_id' => $tenant->id,
                        'user' => $userName,
                        'duration' => $duration,
                    ]);

                    $minutes = round($duration / 60);
                    return [
                        'action' => 'timeout',
                        'message' => "🚫 @{$userName} has been timed out for {$minutes} minutes for spamming.",
                    ];
                }
            }
        }

        return null;
    }

    /**
     * Process moderation checks
     * Returns array with action and message if violation found
     */
    public function moderate(Tenant $tenant, string $text, string $userChannelId, string $userName): ?array
    {
        // Check banned keywords first
        $keywordResult = $this->checkBannedKeywords($tenant, $text, $userChannelId, $userName);
        if ($keywordResult) {
            return $keywordResult;
        }

        // Check spam
        $spamResult = $this->checkSpam($tenant, $text, $userChannelId, $userName);
        if ($spamResult) {
            return $spamResult;
        }

        return null;
    }
}
