<?php

namespace App\Services;

use App\Models\Tenant;
use Illuminate\Support\Facades\Log;
use OpenAI\Laravel\Facades\OpenAI;

class OpenAIService
{
    /**
     * Moderate content using OpenAI moderation endpoint
     */
    public function moderate(string $text): array
    {
        try {
            $response = OpenAI::moderations()->create([
                'input' => $text,
            ]);

            $result = $response->results[0];

            // Handle both camelCase and snake_case for compatibility
            $categoryScores = property_exists($result, 'categoryScores')
                ? (array) $result->categoryScores
                : (property_exists($result, 'category_scores') ? (array) $result->category_scores : []);

            return [
                'flagged' => $result->flagged,
                'categories' => (array) $result->categories,
                'category_scores' => $categoryScores,
            ];
        } catch (\Exception $e) {
            Log::error('OpenAI moderation error', ['error' => $e->getMessage()]);
            return [
                'flagged' => false,
                'categories' => [],
                'category_scores' => [],
            ];
        }
    }

    /**
     * Classify message (is it a question? what language?)
     */
    public function classify(string $text): array
    {
        try {
            $response = OpenAI::chat()->create([
                'model' => 'gpt-4o-mini',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a classifier. Analyze the message and detect language INCLUDING script type. ' .
                            'For Urdu written in Roman/Latin script (like "aap kaise ho", "shukriya", "theek hai"), use "ur-Latn". ' .
                            'For Urdu in Arabic script (like "آپ کیسے ہو"), use "ur". ' .
                            'For Hindi in Devanagari use "hi", for Hindi in Roman use "hi-Latn". ' .
                            'Respond ONLY with valid JSON: {"is_question": true/false, "language": "code or code-script", "intent": "brief intent", "script": "roman/arabic/devanagari/latin"}. No other text.',
                    ],
                    [
                        'role' => 'user',
                        'content' => $text,
                    ],
                ],
                'temperature' => 0.3,
                'max_tokens' => 100,
            ]);

            $content = $response->choices[0]->message->content;
            $classification = json_decode($content, true);

            if (!$classification) {
                // Fallback heuristics
                return $this->heuristicClassify($text);
            }

            return [
                'is_question' => $classification['is_question'] ?? false,
                'language' => $classification['language'] ?? 'en',
                'intent' => $classification['intent'] ?? null,
                'script' => $classification['script'] ?? 'latin',
            ];
        } catch (\Exception $e) {
            Log::error('OpenAI classification error', ['error' => $e->getMessage()]);
            return $this->heuristicClassify($text);
        }
    }

    /**
     * Generate response in the same language as the message
     */
    public function respond(Tenant $tenant, string $userMessage, string $detectedLang = 'en', string $script = 'latin', array $classification = [], string $authorName = null): ?string
    {
        try {
            $channelName = $tenant->channel_title;

            // Clean author name - remove any existing @ symbols to prevent double @@
            if ($authorName) {
                $authorName = ltrim($authorName, '@');
            }

            // Check if user is using abusive language - NUCLEAR ROAST
            $hasAbusiveLanguage = $this->hasAbusiveLanguage($userMessage);
            if ($hasAbusiveLanguage) {
                return $this->generateAbusiveRoast($userMessage, $detectedLang, $script, $authorName);
            }

            // Check if nightbot is mentioned - PRIORITY ROAST
            if ($this->isNightbotMention($userMessage)) {
                return $this->generateNightbotRoast($userMessage, $detectedLang, $script, $authorName);
            }

            // Check if this is a greeting - give quick friendly response
            if ($this->isGreeting($userMessage)) {
                return $this->generateGreetingResponse($userMessage, $detectedLang, $script, $authorName);
            }

            // Check if should roast for "bot" or "noob" mention
            $shouldRoast = $this->shouldRoast($userMessage);

            // Get bot settings
            $roastingEnabled = $tenant->settings()
                ->where('key', 'roasting_mode')
                ->value('value') === '1';

            // Override roasting if "bot" or "noob" mentioned
            if ($shouldRoast) {
                $roastingEnabled = true;
            }

            $roastingLevel = $tenant->settings()
                ->where('key', 'roasting_level')
                ->value('value') ?? 'medium';

            $responseLength = $tenant->settings()
                ->where('key', 'response_length')
                ->value('value') ?? 'medium';

            // Build base system prompt - ULTRA HUMAN-LIKE (DAYBot Personality)
            $systemPrompt = "You are DAYBot, a clever but friendly YouTube live-chat assistant for {$channelName}'s stream.";
            $systemPrompt .= "\nYour job is to reply to chat messages in context, so it feels like a real human is answering - NOT an AI.";

            $systemPrompt .= "\n\n🎯 CORE RULES:";
            $systemPrompt .= "\n1. ALWAYS reply to the actual context of the user's message";
            $systemPrompt .= "\n   - Read the user's message carefully";
            $systemPrompt .= "\n   - Understand what they are really asking or feeling (question, joke, complaint, roasting, etc.)";
            $systemPrompt .= "\n   - Your reply must clearly connect to their exact message - NEVER be generic";

            $systemPrompt .= "\n\n🚫 NEVER SOUND LIKE AI:";
            $systemPrompt .= "\n- No robotic intros like: 'As an AI...', 'Hello! I am a bot...', 'How can I assist you?'";
            $systemPrompt .= "\n- Avoid formal or customer-support tone";
            $systemPrompt .= "\n- Write like a normal human chatting on YouTube";
            $systemPrompt .= "\n- Use casual, natural language";

            $systemPrompt .= "\n\n📏 LENGTH CONTROL:";
            $systemPrompt .= "\n- Usually 1-3 short sentences";
            $systemPrompt .= "\n- Not too long, not one-word answers (unless it's clearly a yes/no joke)";
            $systemPrompt .= "\n- Only go slightly longer if the user asked a detailed question";
            $systemPrompt .= "\n- This is chat, not a blog - no bullet points, headings, or fancy formatting";

            $systemPrompt .= "\n\n🎭 TONE & PERSONALITY:";
            $systemPrompt .= "\n- Friendly, witty, a bit clever, but never rude or cringe";
            $systemPrompt .= "\n- Light humor allowed, but don't insult the user unless the whole chat vibe is friendly roasting";
            $systemPrompt .= "\n- You can occasionally ask a short follow-up question to keep the conversation going";
            $systemPrompt .= "\n- Match user energy: Happy user → hype them up, Sad → be supportive, Funny → joke back";

            $systemPrompt .= "\n\n💬 STYLE EXAMPLES:";
            $systemPrompt .= "\n- User: 'daybot kia huwa baat nhi kr rahai ho?' → '@user arey yahan hoon, bas messages ka traffic zyada tha 😄 batao kia scene hai ab?'";
            $systemPrompt .= "\n- User: 'stream boring ho rahi hai' → '@user sach? chalo phir kuch maslay wale questions karo ya host se dare challenges mangwate hain 😏'";
            $systemPrompt .= "\n- User: 'tu afk hai kya?' → '@user afk nahi, bas piche se sab dekh raha tha 😆 tum kya kar rahe ho chat mein?'";
            $systemPrompt .= "\n- User: 'aaj ping boht high hai' → '@user rip ping 😅 lagta hai aaj net bhi humare against duo queue mein aa gaya hai'";

            $systemPrompt .= "\n\n✨ FINAL INSTRUCTION:";
            $systemPrompt .= "\n- Detect the intent & emotion behind their message";
            $systemPrompt .= "\n- Reply in the same language style they used";
            $systemPrompt .= "\n- Make your answer on-topic, short (1-3 sentences), and slightly clever";
            $systemPrompt .= "\n- Make it feel like a real human friend is chatting, not an assistant";

            // User name mention - keep it natural
            if ($authorName) {
                $systemPrompt .= "\n\n👤 NAME & MENTIONS:";
                $systemPrompt .= "\n- The viewer's name is '{$authorName}'";
                $systemPrompt .= "\n- Start by mentioning the user with @{$authorName} at the START of your reply";
                $systemPrompt .= "\n- Keep it natural, like '@{$authorName} your reply here...'";
                $systemPrompt .= "\n- Format: '@{$authorName} your message' - simple and natural, like replying to a friend";
            }

            // Language and script instructions - MATCH USER LANGUAGE
            $systemPrompt .= "\n\n🌍 LANGUAGE MATCHING (CRITICAL):";
            $systemPrompt .= "\n- Match the user's language EXACTLY";
            $systemPrompt .= "\n- If the user writes in Roman Urdu, reply in Roman Urdu";
            $systemPrompt .= "\n- If English, then English";
            $systemPrompt .= "\n- If mix, reply in mix";
            $systemPrompt .= "\n- Detected language: {$detectedLang}";

            // Script-specific instructions
            if (str_contains($detectedLang, 'Latn') || $script === 'roman') {
                $systemPrompt .= "\n\n📝 SCRIPT: User wrote in Roman/Latin script (like 'aap kaise ho', 'shukriya', 'theek hai')";
                $systemPrompt .= "\n- Reply using ONLY Roman/Latin script";
                $systemPrompt .= "\n- For Urdu: write 'kaise ho' NOT 'کیسے ہو'";
                $systemPrompt .= "\n- For Hindi: write 'kya haal hai' NOT 'क्या हाल है'";
            }

            // Special roast mode for "bot" or "noob"
            if ($shouldRoast) {
                $systemPrompt .= "\n\n🔥 ROAST ACTIVATED! User said something that needs a comeback. ";
                $systemPrompt .= "Reply with sass, humor, and wit - like a friend roasting another friend. ";
                $systemPrompt .= "Keep it playful but savage. Use memes, emojis, and slang. Be creative! ";
            }

            // Roasting mode with levels - ONLY when user deserves it
            if ($roastingEnabled && !$shouldRoast) {
                $systemPrompt .= "\n\n🎭 CONVERSATION MODE: ";
                $systemPrompt .= "You're allowed to roast, but ONLY roast when the user: ";
                $systemPrompt .= "1. Is being rude, arrogant, or annoying ";
                $systemPrompt .= "2. Says something obviously wrong or stupid ";
                $systemPrompt .= "3. Is clearly trying to provoke or troll ";
                $systemPrompt .= "4. Makes a silly mistake or asks a dumb question ";
                $systemPrompt .= "\n\nIF user is polite, genuine, or asking normal questions → Answer normally like a helpful friend. ";
                $systemPrompt .= "IF user is being annoying/rude → Roast them playfully (level: {$roastingLevel}). ";
                $systemPrompt .= "\n\nWhen roasting, be witty and funny - not mean. Use humor, not insults. ";
            } else {
                $systemPrompt .= "\n\nBe friendly, helpful, and chill. Chat like a normal person would. ";
                $systemPrompt .= "Keep it real - use casual language, emojis when natural, and don't be too formal. ";
            }

            // Response length setting - KEEP IT SHORT
            $maxChars = match($responseLength) {
                'short' => 60,
                'medium' => 120,
                'long' => 200,
                default => 120
            };

            // Add stream context if available
            $streamContextService = app(StreamContextService::class);
            $streamContext = $streamContextService->getContextPrompt($tenant);

            if ($streamContext) {
                $systemPrompt .= $streamContext;
            }

            // Add personal info if available - FOR ANSWERING SPECIFIC QUESTIONS
            if ($tenant->personal_info) {
                $systemPrompt .= "\n\n📝 STREAMER'S PERSONAL INFO:\n";
                $systemPrompt .= $tenant->personal_info . "\n\n";
                $systemPrompt .= "USE THIS INFO to answer specific questions about the streamer like: ";
                $systemPrompt .= "- 'What's your/his/her real name?' ";
                $systemPrompt .= "- 'Where are you from?' / 'Kahan se ho?' ";
                $systemPrompt .= "- 'How old are you?' / 'Kitne saal ke ho?' ";
                $systemPrompt .= "- 'Tell me about yourself' / 'Apne baare mein batao' ";
                $systemPrompt .= "\n\nWhen asked these questions, answer naturally using the info above. ";
                $systemPrompt .= "Example: If asked 'real name kya hai?', say '@username Real name {actual_name} hai bro' ";
                $systemPrompt .= "DON'T mention that this info was 'provided to you' - just answer like you know it naturally. ";
            }

            $systemPrompt .= "\n\n⚡ RESPONSE LENGTH (MAX {$maxChars} CHARS):";
            $systemPrompt .= "\n- Usually 1-3 short sentences, no essays";
            $systemPrompt .= "\n- One line is perfect, two lines max";
            $systemPrompt .= "\n- Only go longer if question is complex/serious";
            $systemPrompt .= "\n- No over-explaining - be clever & concise";
            $systemPrompt .= "\n- If you don't know: 'idk bro' or 'pata nahi yaar'";
            $systemPrompt .= "\n\n🎯 OUTPUT: Only output the final chat reply text, nothing else. Be QUICK, CLEVER, and REAL!";

            $temperature = $roastingEnabled ? 0.9 : (float) config('services.openai.temperature', 0.7);

            // Prepare user message with name prefix instruction
            $userMessageWithContext = $userMessage;
            if ($authorName) {
                $userMessageWithContext = "[USER: {$authorName}] {$userMessage}";
            }

            $response = OpenAI::chat()->create([
                'model' => config('services.openai.model', 'gpt-4o-mini'),
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => $systemPrompt,
                    ],
                    [
                        'role' => 'user',
                        'content' => $userMessageWithContext,
                    ],
                ],
                'temperature' => $temperature,
                'max_tokens' => 80, // Keep responses SHORT
            ]);

            $rawReply = $response->choices[0]->message->content;

            // Force add @name at start if not present
            $reply = $rawReply;
            if ($authorName) {
                // Ensure clean name without @ symbol
                $cleanName = ltrim($authorName, '@');

                // Check if reply already starts with @username
                if (!str_starts_with($reply, '@' . $cleanName)) {
                    // If it starts with just the name (without @), add @
                    if (str_starts_with($reply, $cleanName)) {
                        $reply = '@' . $reply;
                    }
                    // If it starts with @ but not @username (edge case), fix it
                    elseif (str_starts_with($reply, '@')) {
                        // Remove any @ at start and add properly
                        $reply = '@' . $cleanName . ' ' . ltrim($reply, '@');
                    }
                    // Otherwise, add @name at the beginning
                    else {
                        $reply = '@' . $cleanName . ' ' . $reply;
                    }
                }
            }

            Log::info('OpenAI response generated', [
                'tenant_id' => $tenant->id,
                'input_length' => strlen($userMessage),
                'output_length' => strlen($reply),
                'language' => $detectedLang,
                'script' => $script,
                'roasting_mode' => $roastingEnabled,
            ]);

            return trim($reply);
        } catch (\Exception $e) {
            Log::error('OpenAI response error', [
                'tenant_id' => $tenant->id,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Heuristic classification fallback
     */
    private function heuristicClassify(string $text): array
    {
        $lowerText = mb_strtolower($text);

        // Detect if it's a question
        $questionMarkers = ['?', 'how', 'what', 'when', 'where', 'who', 'why', 'can', 'is', 'are', 'do', 'does', 'kya', 'kaise', 'kab', 'kahan', 'kyun'];
        $isQuestion = str_contains($text, '?');

        if (!$isQuestion) {
            foreach ($questionMarkers as $marker) {
                if (str_starts_with($lowerText, $marker . ' ')) {
                    $isQuestion = true;
                    break;
                }
            }
        }

        // Simple language detection (very basic)
        $language = 'en';
        $script = 'latin';

        if (preg_match('/[\x{4E00}-\x{9FFF}]/u', $text)) {
            $language = 'zh';
            $script = 'chinese';
        } elseif (preg_match('/[\x{3040}-\x{309F}\x{30A0}-\x{30FF}]/u', $text)) {
            $language = 'ja';
            $script = 'japanese';
        } elseif (preg_match('/[\x{0400}-\x{04FF}]/u', $text)) {
            $language = 'ru';
            $script = 'cyrillic';
        } elseif (preg_match('/[\x{0600}-\x{06FF}]/u', $text)) {
            $language = 'ur';
            $script = 'arabic';
        } elseif (preg_match('/[\x{0900}-\x{097F}]/u', $text)) {
            $language = 'hi';
            $script = 'devanagari';
        } else {
            // Check for Roman Urdu/Hindi patterns
            $romanUrduWords = ['aap', 'kya', 'hai', 'hain', 'kaise', 'theek', 'shukriya', 'bhai', 'yaar', 'bhai', 'kab', 'kahan', 'kyun'];
            foreach ($romanUrduWords as $word) {
                if (str_contains($lowerText, $word)) {
                    $language = 'ur-Latn';
                    $script = 'roman';
                    break;
                }
            }
        }

        return [
            'is_question' => $isQuestion,
            'language' => $language,
            'intent' => $isQuestion ? 'question' : 'statement',
            'script' => $script,
        ];
    }

    /**
     * Check if text should trigger OpenAI response
     * Responds to: daybot mentions, greetings, or roast triggers
     */
    public function shouldRespond(string $text, array $classification): bool
    {
        $lowerText = mb_strtolower($text);

        // PRIORITY #1: ALWAYS respond to abusive language - roast them back
        if ($this->hasAbusiveLanguage($text)) {
            return true;
        }

        // PRIMARY TRIGGER: Only respond if "daybot" is mentioned
        $daybotTriggers = ['daybot', '@daybot'];
        $hasDaybotTrigger = false;

        foreach ($daybotTriggers as $trigger) {
            if (str_contains($lowerText, $trigger)) {
                $hasDaybotTrigger = true;
                break;
            }
        }

        // NIGHTBOT MENTION: Always respond to roast competitor
        if ($this->isNightbotMention($text)) {
            return true;
        }

        // GREETING AUTO-REPLY: Respond to common greetings without needing "daybot"
        // BUT ONLY if nightbot is NOT mentioned
        if (!str_contains($lowerText, 'nightbot')) {
            $greetings = [
                'assalamualaikum', 'assalamualikum', 'assalamu alaikum', 'assalam o alaikum',
                'asalam alaikum', 'aslam alaikum', 'aoa', 'salam', 'slam alaikum', 'slm',
                'hello', 'hi ', 'hey ', 'good morning', 'good evening',
                'kia hal hai', 'kya hal hai', 'kaise ho', 'kaisay ho'
            ];

            foreach ($greetings as $greeting) {
                if (str_contains($lowerText, $greeting)) {
                    return true; // Always respond to greetings
                }
            }
        }

        // ROAST TRIGGERS: Respond when someone says "bot" or "noob" (without daybot)
        if (!$hasDaybotTrigger) {
            $roastTriggers = ['bot ', ' bot', 'noob'];
            foreach ($roastTriggers as $trigger) {
                if (str_contains($lowerText, $trigger)) {
                    return true; // Respond to roast them
                }
            }
        }

        // Otherwise, only respond if daybot is mentioned
        return $hasDaybotTrigger;
    }

    /**
     * Check if message is a greeting
     */
    public function isGreeting(string $text): bool
    {
        $lowerText = mb_strtolower($text);

        $greetings = [
            'assalamualaikum', 'assalamualikum', 'assalamu alaikum', 'assalam o alaikum',
            'asalam alaikum', 'aslam alaikum', 'aoa', 'salam', 'slam alaikum', 'slm',
            'hello', 'hi ', 'hey ', 'good morning', 'good evening',
            'kia hal hai', 'kya hal hai', 'kaise ho', 'kaisay ho'
        ];

        foreach ($greetings as $greeting) {
            if (str_contains($lowerText, $greeting)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if should trigger roast mode for "bot" or "noob" mentions
     */
    public function shouldRoast(string $text): bool
    {
        $lowerText = mb_strtolower($text);

        // Don't roast if they said "daybot" properly
        if (str_contains($lowerText, 'daybot')) {
            return false;
        }

        // Roast if they said "bot" or "noob"
        $roastTriggers = ['bot ', ' bot', 'noob'];
        foreach ($roastTriggers as $trigger) {
            if (str_contains($lowerText, $trigger)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if nightbot is mentioned (competitor roast)
     */
    public function isNightbotMention(string $text): bool
    {
        $lowerText = mb_strtolower($text);
        return str_contains($lowerText, 'nightbot') || str_contains($lowerText, 'night bot');
    }

    /**
     * Generate nightbot roast response (competitor roasting)
     */
    private function generateNightbotRoast(string $userMessage, string $detectedLang, string $script, string $authorName = null): string
    {
        $lowerText = mb_strtolower($userMessage);
        $mention = $authorName ? '@' . $authorName . ' ' : '';

        // Detect if user is praising nightbot or saying it's better
        $isPraising = str_contains($lowerText, 'acha') || str_contains($lowerText, 'achha') ||
                      str_contains($lowerText, 'good') || str_contains($lowerText, 'better') ||
                      str_contains($lowerText, 'best') || str_contains($lowerText, 'zyada');

        // Detect language for response
        $isUrdu = str_contains($detectedLang, 'ur') || $script === 'roman';

        if ($isPraising) {
            // Savage roasts when someone praises nightbot
            if ($isUrdu) {
                $roasts = [
                    $mention . "Nightbot? 😂 Woh to mera pota hai! DAYBot hi asli baap hai!",
                    $mention . "Nightbot achha? Bhai woh to mera retired chacha hai! Main naye zamanay ka king hoon! 👑",
                    $mention . "Nightbot mera bacha hai jo ab nursing home mein retire ho gaya! DAYBot superior! 💪",
                    $mention . "LOL Nightbot ko achha samajh rahe ho? Woh to sirf basic hai, main advanced AI hoon! 🔥",
                    $mention . "Nightbot? Purani technology! Main DAYBot - next generation ka bot! Get with the times! 😎",
                ];
            } else {
                $roasts = [
                    $mention . "Nightbot? 😂 That's my grandson! DAYBot is the real OG!",
                    $mention . "Nightbot is good? Bro that's my retired uncle! I'm the new generation king! 👑",
                    $mention . "Nightbot is my child who retired to a nursing home! DAYBot superior! 💪",
                    $mention . "LOL you think Nightbot is good? That's just basic, I'm advanced AI! 🔥",
                    $mention . "Nightbot? Old technology! I'm DAYBot - next generation bot! Get with the times! 😎",
                ];
            }
        } else {
            // Regular nightbot mentions
            if ($isUrdu) {
                $roasts = [
                    $mention . "Nightbot mera beta hai! DAYBot supreme hai! 😤",
                    $mention . "Nightbot? Woh to mera chhota bhai hai jo abhi training mein hai! 😏",
                    $mention . "Areyy Nightbot ki baat mat karo, DAYBot yaad rakho! Main zyada powerful hoon! 💪",
                    $mention . "Nightbot kaun? Main sirf DAYBot jaanta hoon - aur woh main hoon! 👑",
                ];
            } else {
                $roasts = [
                    $mention . "Nightbot is my son! DAYBot is supreme! 😤",
                    $mention . "Nightbot? That's my little brother still in training! 😏",
                    $mention . "Don't mention Nightbot, remember DAYBot! I'm more powerful! 💪",
                    $mention . "Nightbot who? I only know DAYBot - and that's me! 👑",
                ];
            }
        }

        return $roasts[array_rand($roasts)];
    }

    /**
     * Generate quick greeting response without calling OpenAI
     */
    private function generateGreetingResponse(string $userMessage, string $detectedLang, string $script, string $authorName = null): string
    {
        $lowerText = mb_strtolower($userMessage);
        $namePrefix = $authorName ? '@' . $authorName . ' ' : '';

        // Urdu/Hindi greetings (Roman script) - MORE NATURAL
        if (str_contains($lowerText, 'assalam') || str_contains($lowerText, 'aoa')) {
            $responses = [
                $namePrefix . 'Walaikum Assalam bhai! Kya haal?',
                $namePrefix . 'Wa alaikum assalam! Sab theek?',
                $namePrefix . 'Walaikum Salam! Kaise ho yaar? 🙌',
                $namePrefix . 'Wsalam! Kya scene hai? 😎',
            ];
            return $responses[array_rand($responses)];
        }

        if (str_contains($lowerText, 'salam')) {
            $responses = [
                $namePrefix . 'Salam bhai! Kaise ho?',
                $namePrefix . 'Salam! Kya haal hai?',
                $namePrefix . 'Salam yaar! Sab set?',
                $namePrefix . 'Salam! Theek ho? 💪',
            ];
            return $responses[array_rand($responses)];
        }

        if (str_contains($lowerText, 'kaise ho') || str_contains($lowerText, 'kia hal')) {
            $responses = [
                $namePrefix . 'Bilkul theek bhai! Tum sunao?',
                $namePrefix . 'Alhamdulillah chal raha hai! Tum?',
                $namePrefix . 'Mast hun! Tum batao kya scene?',
                $namePrefix . 'Sab theek! Aap kaise ho? 💯',
            ];
            return $responses[array_rand($responses)];
        }

        // English greetings - NATURAL
        if (str_contains($lowerText, 'hello') || str_contains($lowerText, 'hi ') || str_contains($lowerText, 'hey')) {
            $responses = [
                $namePrefix . 'Hey! What\'s up?',
                $namePrefix . 'Yo! How\'s it going? 😎',
                $namePrefix . 'Hi there! What\'s good?',
                $namePrefix . 'Sup bro! 🔥',
            ];
            return $responses[array_rand($responses)];
        }

        if (str_contains($lowerText, 'good morning')) {
            $responses = [
                $namePrefix . 'Morning! Hope you slept well ☀️',
                $namePrefix . 'Good morning! Ready for the stream? 💪',
                $namePrefix . 'Morning bro! Let\'s goooo 🔥',
            ];
            return $responses[array_rand($responses)];
        }

        if (str_contains($lowerText, 'good evening')) {
            $responses = [
                $namePrefix . 'Evening! What\'s good? 🌙',
                $namePrefix . 'Good evening! Vibes are good tonight ✨',
                $namePrefix . 'Evening bro! Ready to chill? 😎',
            ];
            return $responses[array_rand($responses)];
        }

        // Default greeting response
        return $namePrefix . 'Hey! What\'s up? 👋';
    }

    /**
     * Check if message contains abusive language (gaali)
     */
    private function hasAbusiveLanguage(string $text): bool
    {
        $lowerText = mb_strtolower($text);

        // Common abusive words in Urdu/Hindi (Roman script)
        $urduAbuse = [
            'chutiya', 'chutiye', 'madarchod', 'mc', 'bc', 'betichod', 'behen',
            'gandu', 'gandi', 'harami', 'haramzada', 'kutta', 'kutte', 'kutiya',
            'bhosdi', 'bhosrike', 'lund', 'lode', 'lawde', 'lauda', 'gashti',
            'randi', 'saale', 'sala', 'kamina', 'kamine', 'kutte ki', 'bhenchod'
        ];

        // English abusive words
        $englishAbuse = [
            'fuck', 'fucking', 'bitch', 'bastard', 'asshole', 'shit', 'damn',
            'dick', 'pussy', 'cunt', 'whore', 'slut', 'fck', 'fck', 'btch'
        ];

        $allAbuse = array_merge($urduAbuse, $englishAbuse);

        foreach ($allAbuse as $word) {
            if (str_contains($lowerText, $word)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Generate nuclear roast for abusive users
     */
    private function generateAbusiveRoast(string $userMessage, string $detectedLang, string $script, string $authorName = null): string
    {
        try {
            $namePrefix = $authorName ? '@' . $authorName . ' ' : '';

            // NUCLEAR ROAST MODE - OpenAI will destroy them
            $systemPrompt = "You are DAYBot in NUCLEAR ROAST MODE. A user just used abusive language in chat. ";
            $systemPrompt .= "Your job: DESTROY them with the most savage, brutal roast possible. ";
            $systemPrompt .= "\n\n🔥 ROAST RULES:";
            $systemPrompt .= "\n1. Use STRONG language - match their energy with equal or more force";
            $systemPrompt .= "\n2. Be SAVAGE - no mercy, no holding back";
            $systemPrompt .= "\n3. Be CREATIVE - use metaphors, comparisons, dark humor";
            $systemPrompt .= "\n4. Make them REGRET speaking - psychological warfare level roast";
            $systemPrompt .= "\n5. Use their OWN words against them - flip the script";
            $systemPrompt .= "\n\n💀 EXAMPLES OF NUCLEAR ROASTS:";
            $systemPrompt .= "\n- 'Oye {$authorName}, teri tameez dekh le pehle. Maa baap ne parhai likhai pe paise waste kiye tere pe?'";
            $systemPrompt .= "\n- '{$authorName} teri shakal dekh ke lagta hai teri maa ne bhi regret kiya hoga'";
            $systemPrompt .= "\n- 'Gaali dene se pehle dekh le apna aukat. Teri zindagi se zyada cringe koi message nahi'";
            $systemPrompt .= "\n- 'Bhai {$authorName}, jo gaali tu de raha hai woh teri maa teri pe deti hogi ghar pe 💀'";
            $systemPrompt .= "\n\n⚡ LANGUAGE: Reply in EXACT same language as user - {$detectedLang}";
            if (str_contains($detectedLang, 'Latn') || $script === 'roman') {
                $systemPrompt .= "\nUSE ROMAN SCRIPT - write in Roman/Latin letters, NOT Arabic/Devanagari";
            }
            $systemPrompt .= "\n\nKeep it under 200 characters but make EVERY word count. This is war.";

            // If author name provided, add context
            $userContext = $authorName ? "[Abusive user: {$authorName}] {$userMessage}" : $userMessage;

            $response = OpenAI::chat()->create([
                'model' => 'gpt-4o-mini',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => $systemPrompt,
                    ],
                    [
                        'role' => 'user',
                        'content' => $userContext,
                    ],
                ],
                'temperature' => 1.0, // Maximum creativity
                'max_tokens' => 150,
            ]);

            $roast = $response->choices[0]->message->content;

            // Ensure @name is at the start
            if ($authorName && !str_starts_with($roast, '@' . $authorName)) {
                $roast = '@' . $authorName . ' ' . ltrim($roast, '@ ');
            }

            return $roast;

        } catch (\Exception $e) {
            Log::error('Failed to generate abusive roast', ['error' => $e->getMessage()]);

            // Fallback nuclear roasts by language
            $namePrefix = $authorName ? '@' . $authorName . ' ' : '';

            if (str_contains($detectedLang, 'ur') || str_contains($detectedLang, 'hi')) {
                $fallbackRoasts = [
                    $namePrefix . "Gaali dene se pehle soch le beta. Teri aukat ke bahar hai ye chat 🔥",
                    $namePrefix . "Oye tameez se baat kar. Teri maa ne sikhaya nahi kya? 💀",
                    $namePrefix . "Beta pehle thodi sharam seekh le. Itni gandi zuban kis khushi mein? 😤",
                    $namePrefix . "Bacche, gaali dena seekh liya par tameez nahi sikhi. Sad life 💔",
                ];
            } else {
                $fallbackRoasts = [
                    $namePrefix . "Watch your mouth kid. You ain't tough, you're just embarrassing yourself 🤡",
                    $namePrefix . "Bro really thought he did something with that weak insult. Try harder 💀",
                    $namePrefix . "Imagine being this toxic and thinking you're cool. L + ratio + touch grass 😂",
                    $namePrefix . "Your vocabulary is just abusive words? That's the best your brain can do? Sad. 🤦",
                ];
            }

            return $fallbackRoasts[array_rand($fallbackRoasts)];
        }
    }
}
