<?php

namespace App\Services;

use App\Models\Tenant;
use Illuminate\Support\Facades\Cache;

class RateLimiterService
{
    private const OUTBOUND_PER_MIN_KEY = 'rate_limit:outbound:min:';
    private const OUTBOUND_PER_HOUR_KEY = 'rate_limit:outbound:hour:';

    /**
     * Check if tenant can send outbound message
     */
    public function allowOutbound(Tenant $tenant): bool
    {
        // Get tenant-specific quotas or use defaults
        $quotas = $tenant->quotas ?? [];
        $perMinLimit = $quotas['per_minute'] ?? config('services.bot.rate_limit_outbound_per_min', 200);
        $perHourLimit = $quotas['per_hour'] ?? config('services.bot.rate_limit_outbound_per_hour', 2000);

        // Check per-minute limit
        $minKey = self::OUTBOUND_PER_MIN_KEY . $tenant->id;
        $minCount = (int) Cache::get($minKey, 0);

        if ($minCount >= $perMinLimit) {
            return false;
        }

        // Check per-hour limit
        $hourKey = self::OUTBOUND_PER_HOUR_KEY . $tenant->id;
        $hourCount = (int) Cache::get($hourKey, 0);

        if ($hourCount >= $perHourLimit) {
            return false;
        }

        return true;
    }

    /**
     * Record outbound message
     */
    public function recordOutbound(Tenant $tenant): void
    {
        $minKey = self::OUTBOUND_PER_MIN_KEY . $tenant->id;
        $hourKey = self::OUTBOUND_PER_HOUR_KEY . $tenant->id;

        // Increment per-minute counter (expires in 60 seconds)
        if (!Cache::has($minKey)) {
            Cache::put($minKey, 1, 60);
        } else {
            Cache::increment($minKey);
        }

        // Increment per-hour counter (expires in 1 hour)
        if (!Cache::has($hourKey)) {
            Cache::put($hourKey, 1, 3600);
        } else {
            Cache::increment($hourKey);
        }
    }

    /**
     * Get remaining quota for tenant
     */
    public function getRemainingQuota(Tenant $tenant): array
    {
        // Get tenant-specific quotas or use defaults
        $quotas = $tenant->quotas ?? [];
        $perMinLimit = $quotas['per_minute'] ?? config('services.bot.rate_limit_outbound_per_min', 200);
        $perHourLimit = $quotas['per_hour'] ?? config('services.bot.rate_limit_outbound_per_hour', 2000);

        $minKey = self::OUTBOUND_PER_MIN_KEY . $tenant->id;
        $hourKey = self::OUTBOUND_PER_HOUR_KEY . $tenant->id;

        $minCount = (int) Cache::get($minKey, 0);
        $hourCount = (int) Cache::get($hourKey, 0);

        return [
            'per_minute' => [
                'limit' => $perMinLimit,
                'used' => $minCount,
                'remaining' => max(0, $perMinLimit - $minCount),
            ],
            'per_hour' => [
                'limit' => $perHourLimit,
                'used' => $hourCount,
                'remaining' => max(0, $perHourLimit - $hourCount),
            ],
        ];
    }

    /**
     * Reset rate limits for tenant (admin override)
     */
    public function reset(Tenant $tenant): void
    {
        $minKey = self::OUTBOUND_PER_MIN_KEY . $tenant->id;
        $hourKey = self::OUTBOUND_PER_HOUR_KEY . $tenant->id;

        Cache::forget($minKey);
        Cache::forget($hourKey);
    }
}
