<?php

namespace App\Services;

use App\Models\Tenant;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use OpenAI\Laravel\Facades\OpenAI;

class StreamContextService
{
    /**
     * Detect and update stream context for a tenant
     */
    public function detectAndUpdateContext(Tenant $tenant): bool
    {
        try {
            // Get current live stream or latest video details
            $videoDetails = $this->getVideoDetails($tenant);

            if (!$videoDetails) {
                Log::warning('Could not fetch video details', ['tenant_id' => $tenant->id]);
                return false;
            }

            // Use OpenAI to analyze and classify stream type
            $context = $this->analyzeStreamContext($videoDetails);

            if ($context) {
                // Update tenant with detected context
                $tenant->update([
                    'stream_type' => $context['stream_type'],
                    'current_game' => $context['current_game'] ?? null,
                    'stream_description' => $context['description'] ?? null,
                    'stream_context_updated_at' => now(),
                ]);

                Log::info('Stream context updated', [
                    'tenant_id' => $tenant->id,
                    'stream_type' => $context['stream_type'],
                    'current_game' => $context['current_game'] ?? 'N/A',
                ]);

                return true;
            }

            return false;
        } catch (\Exception $e) {
            Log::error('Stream context detection failed', [
                'tenant_id' => $tenant->id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Get current live stream or latest video details from YouTube
     */
    private function getVideoDetails(Tenant $tenant): ?array
    {
        try {
            $apiKey = config('services.google.api_key');
            if (!$apiKey) {
                Log::error('Google API key not configured');
                return null;
            }

            // First, try to get current live stream
            $searchUrl = 'https://www.googleapis.com/youtube/v3/search';
            $response = Http::get($searchUrl, [
                'part' => 'snippet',
                'channelId' => $tenant->channel_id,
                'eventType' => 'live',
                'type' => 'video',
                'maxResults' => 1,
                'key' => $apiKey,
            ]);

            $videoId = null;

            if ($response->successful()) {
                $data = $response->json();
                if (!empty($data['items'])) {
                    $videoId = $data['items'][0]['id']['videoId'];
                }
            }

            // If no live stream, get latest video
            if (!$videoId) {
                $response = Http::get($searchUrl, [
                    'part' => 'snippet',
                    'channelId' => $tenant->channel_id,
                    'order' => 'date',
                    'type' => 'video',
                    'maxResults' => 1,
                    'key' => $apiKey,
                ]);

                if ($response->successful()) {
                    $data = $response->json();
                    if (!empty($data['items'])) {
                        $videoId = $data['items'][0]['id']['videoId'];
                    }
                }
            }

            if (!$videoId) {
                Log::warning('No videos found for channel', ['channel_id' => $tenant->channel_id]);
                return null;
            }

            // Get full video details
            $videoUrl = 'https://www.googleapis.com/youtube/v3/videos';
            $response = Http::get($videoUrl, [
                'part' => 'snippet,liveStreamingDetails',
                'id' => $videoId,
                'key' => $apiKey,
            ]);

            if (!$response->successful()) {
                Log::error('Failed to get video details', [
                    'video_id' => $videoId,
                    'response' => $response->json(),
                ]);
                return null;
            }

            $data = $response->json();
            if (empty($data['items'])) {
                return null;
            }

            $video = $data['items'][0];
            $snippet = $video['snippet'];

            return [
                'video_id' => $videoId,
                'title' => $snippet['title'] ?? '',
                'description' => $snippet['description'] ?? '',
                'category_id' => $snippet['categoryId'] ?? '',
                'tags' => $snippet['tags'] ?? [],
                'is_live' => !empty($video['liveStreamingDetails']),
            ];
        } catch (\Exception $e) {
            Log::error('YouTube API error', [
                'tenant_id' => $tenant->id,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Use OpenAI to analyze video details and classify stream type
     */
    private function analyzeStreamContext(array $videoDetails): ?array
    {
        try {
            $prompt = "Analyze this YouTube video/stream and classify it:\n\n";
            $prompt .= "Title: {$videoDetails['title']}\n";
            $prompt .= "Description: " . substr($videoDetails['description'], 0, 500) . "\n";

            if (!empty($videoDetails['tags'])) {
                $prompt .= "Tags: " . implode(', ', array_slice($videoDetails['tags'], 0, 10)) . "\n";
            }

            $prompt .= "\nClassify this content into one of these stream types:\n";
            $prompt .= "- gaming (if playing video games)\n";
            $prompt .= "- tech (technology, coding, tutorials)\n";
            $prompt .= "- music (singing, instruments, DJ)\n";
            $prompt .= "- cooking (food, recipes)\n";
            $prompt .= "- fitness (workout, exercise)\n";
            $prompt .= "- vlog (daily life, talking)\n";
            $prompt .= "- education (teaching, learning)\n";
            $prompt .= "- comedy (jokes, funny content)\n";
            $prompt .= "- art (drawing, painting, crafts)\n";
            $prompt .= "- sports (watching/discussing sports)\n";
            $prompt .= "- other (anything else)\n\n";

            $prompt .= "If gaming, also identify the EXACT game name (e.g., 'GTA V', 'Valorant', 'Minecraft').\n\n";
            $prompt .= "Respond ONLY with valid JSON:\n";
            $prompt .= '{"stream_type": "type", "current_game": "game name or null", "description": "brief 1-line description"}';

            $response = OpenAI::chat()->create([
                'model' => 'gpt-4o-mini',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a stream classifier. Respond only with valid JSON, no other text.',
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt,
                    ],
                ],
                'temperature' => 0.3,
                'max_tokens' => 150,
            ]);

            $content = $response->choices[0]->message->content;

            // Remove markdown code blocks if present
            $content = preg_replace('/```json\s*/', '', $content);
            $content = preg_replace('/```\s*/', '', $content);
            $content = trim($content);

            $result = json_decode($content, true);

            if (!$result || !isset($result['stream_type'])) {
                Log::warning('Invalid OpenAI classification response', ['response' => $content]);
                return null;
            }

            return [
                'stream_type' => $result['stream_type'],
                'current_game' => $result['current_game'] ?? null,
                'description' => $result['description'] ?? null,
            ];
        } catch (\Exception $e) {
            Log::error('OpenAI stream classification error', [
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Get stream context for OpenAI prompt
     */
    public function getContextPrompt(Tenant $tenant): string
    {
        $streamType = $tenant->stream_type;
        $currentGame = $tenant->current_game;
        $description = $tenant->stream_description;

        if (!$streamType) {
            return '';
        }

        $prompt = "\n\n📺 STREAM CONTEXT:\n";
        $prompt .= "This is a {$streamType} stream/channel. ";

        if ($streamType === 'gaming' && $currentGame) {
            $prompt .= "Currently playing: {$currentGame}. ";
            $prompt .= "Make references to {$currentGame} gameplay, mechanics, and community in your responses and roasts. ";
        }

        if ($description) {
            $prompt .= "Stream info: {$description}. ";
        }

        $prompt .= "Tailor your responses to match this content type. ";
        $prompt .= "Use relevant terminology, jokes, and references specific to {$streamType}";

        if ($currentGame) {
            $prompt .= " and {$currentGame}";
        }

        $prompt .= ". ";

        return $prompt;
    }

    /**
     * Should we refresh context? (every 30 minutes for live streams)
     */
    public function shouldRefreshContext(Tenant $tenant): bool
    {
        if (!$tenant->stream_context_updated_at) {
            return true;
        }

        // Refresh context every 30 minutes
        return $tenant->stream_context_updated_at->diffInMinutes(now()) >= 30;
    }
}
