# 🚀 Quick Start Guide - PXEBot Design System

## Get Running in 3 Minutes!

---

## ⚡ Installation

### Step 1: Install Dependencies
```bash
cd C:\xampp\htdocs\phxbot.com\nextjs-redesign
npm install
```

This will install:
- Next.js 14
- React 18
- TailwindCSS 3.4
- PostCSS & Autoprefixer

---

### Step 2: Start Development Server
```bash
npm run dev
```

Open your browser to: **http://localhost:3000**

---

## 📄 Available Pages

Once running, visit these URLs:

### Public Pages (No Auth Required)
- **Landing Page**: http://localhost:3000/
- **Login**: http://localhost:3000/login
- **Register**: http://localhost:3000/register
- **404 Test**: http://localhost:3000/random-page

### Dashboard Pages (Mock Auth)
- **Dashboard**: http://localhost:3000/dashboard
- **Commands**: http://localhost:3000/commands
- **Timers**: http://localhost:3000/timers
- **Settings**: http://localhost:3000/settings
- **Billing**: http://localhost:3000/billing

---

## 🎨 Design System Reference

### Colors (Use in Tailwind)
```jsx
// Backgrounds
bg-background      // #0B0F19
bg-surface         // #111827
border-border      // #1E293B

// Brand Colors
text-primary       // #3B82F6
text-secondary     // #38BDF8
bg-primary         // Blue
bg-secondary       // Cyan

// Status Colors
bg-success         // Green
bg-warning         // Yellow
bg-danger          // Red

// Text
text-text-primary      // White
text-text-secondary    // Gray
text-text-muted        // Dark Gray
```

---

### Component Usage Examples

#### Button
```jsx
import Button from '@/components/ui/Button';

// Primary CTA
<Button variant="primary" size="lg" fullWidth>
  Get Started
</Button>

// Secondary Action
<Button variant="secondary" size="md">
  Learn More
</Button>

// Danger Action
<Button variant="danger" size="sm">
  Delete
</Button>

// With Loading State
<Button variant="primary" loading={isLoading}>
  Save Changes
</Button>

// With Icon
<Button variant="primary" icon={<PlusIcon />}>
  Add Channel
</Button>
```

---

#### Card
```jsx
import Card from '@/components/ui/Card';

// Simple Card
<Card padding="default">
  Content here
</Card>

// With Title & Subtitle
<Card
  title="Profile Settings"
  subtitle="Manage your account information"
  padding="default"
>
  Form content here
</Card>

// With Icon & Action
<Card
  title="Bot Status"
  icon={<BotIcon />}
  action={<Button>Edit</Button>}
  hover
>
  Status content
</Card>
```

---

#### Input
```jsx
import Input from '@/components/ui/Input';

// Basic Input
<Input
  label="Email"
  type="email"
  placeholder="you@example.com"
/>

// With Icon
<Input
  label="Password"
  type="password"
  icon={<LockIcon />}
  hint="At least 8 characters"
/>

// With Error
<Input
  label="Username"
  error="Username already taken"
/>
```

---

#### StatCard
```jsx
import StatCard from '@/components/ui/StatCard';

<StatCard
  label="Messages / 1min"
  value="142"
  subtitle="last minute"
  color="primary"
  trend="up"
  trendValue="+12%"
  icon={<ChartIcon />}
/>
```

---

#### Badge
```jsx
import Badge from '@/components/ui/Badge';

<Badge variant="success" size="md">
  Active
</Badge>

<Badge variant="danger" size="sm">
  Error
</Badge>

<Badge variant="primary">
  New Feature
</Badge>
```

---

#### Toggle
```jsx
import Toggle from '@/components/ui/Toggle';

const [enabled, setEnabled] = useState(false);

<Toggle
  label="Enable Notifications"
  description="Get alerts when bot status changes"
  checked={enabled}
  onChange={setEnabled}
/>
```

---

## 🏗️ Creating a New Page

### Example: Creating a "Profile" Page

1. **Create File**: `app/profile/page.jsx`

```jsx
'use client';

import React from 'react';
import Layout from '@/components/layout/Layout';
import Card from '@/components/ui/Card';
import Button from '@/components/ui/Button';

const mockUser = {
  name: 'Your Name',
  email: 'you@example.com',
};

export default function Profile() {
  return (
    <Layout user={mockUser}>
      {/* Page Header */}
      <div className="mb-8">
        <h1 className="text-title text-text-primary">My Profile</h1>
        <p className="text-text-secondary">
          Manage your personal information
        </p>
      </div>

      {/* Content */}
      <Card title="Profile Information">
        <p className="text-text-secondary">
          Your profile content here
        </p>
        <div className="mt-6">
          <Button variant="primary">
            Save Changes
          </Button>
        </div>
      </Card>
    </Layout>
  );
}
```

2. **Visit**: http://localhost:3000/profile

---

## 🎨 Customizing the Design

### Change Primary Color

Edit `tailwind.config.js`:

```js
colors: {
  primary: {
    DEFAULT: '#8B5CF6', // Change to purple
    light: '#A78BFA',
    dark: '#7C3AED',
  },
}
```

### Change Font

Edit `app/layout.jsx`:

```js
import { Poppins } from 'next/font/google';

const poppins = Poppins({
  weight: ['400', '500', '600', '700'],
  subsets: ['latin'],
});
```

### Change Max Width

Edit `tailwind.config.js`:

```js
maxWidth: {
  'container': '1200px', // Changed from 1440px
}
```

---

## 🚀 Building for Production

### Build
```bash
npm run build
```

### Start Production Server
```bash
npm start
```

### Deploy to Vercel
```bash
# Install Vercel CLI
npm i -g vercel

# Deploy
vercel
```

---

## 📱 Testing Responsive Design

### Browser DevTools
1. Open DevTools (F12)
2. Click "Toggle Device Toolbar" (Ctrl+Shift+M)
3. Test these sizes:
   - Mobile: 375px (iPhone)
   - Tablet: 768px (iPad)
   - Desktop: 1440px (Large screen)

### Breakpoints to Test
- **sm**: 640px - Small tablets
- **md**: 768px - Tablets
- **lg**: 1024px - Laptops
- **xl**: 1280px - Desktops

---

## 🐛 Common Issues

### Port 3000 Already in Use
```bash
# Kill process on port 3000 (Windows)
netstat -ano | findstr :3000
taskkill /PID <PID> /F

# Or use different port
npm run dev -- -p 3001
```

### Tailwind Classes Not Working
```bash
# Restart dev server
Ctrl+C
npm run dev
```

### Module Not Found
```bash
# Reinstall dependencies
rm -rf node_modules package-lock.json
npm install
```

---

## 📚 Next Steps

1. **Explore Pages**: Visit all pages to see the design
2. **Read Components**: Check `components/ui/` folder
3. **Customize**: Edit colors in `tailwind.config.js`
4. **Build Pages**: Create new pages following examples
5. **Integrate API**: Connect to your backend

---

## 🎯 Key Files to Know

```
nextjs-redesign/
├── app/
│   ├── layout.jsx              # Root layout (font, metadata)
│   ├── globals.css             # Global styles
│   └── [page]/page.jsx         # Individual pages
├── components/
│   ├── ui/                     # Reusable components
│   └── layout/                 # Layout components
├── tailwind.config.js          # Tailwind customization
└── jsconfig.json               # Path aliases (@/)
```

---

## 💡 Pro Tips

1. **Use Path Aliases**: `@/components/ui/Button` instead of `../../components/ui/Button`
2. **Follow Naming**: Keep component names PascalCase
3. **Use Variants**: Don't create new styles, use existing variants
4. **Check Mobile**: Always test on mobile view
5. **Consistent Spacing**: Use gap-4, gap-6, gap-8 (not gap-5)

---

## 🆘 Need Help?

- **README.md**: Full documentation
- **DESIGN_SYSTEM_COMPLETE.md**: Complete overview
- **Component Files**: Check comments in code
- **TailwindCSS Docs**: https://tailwindcss.com/docs
- **Next.js Docs**: https://nextjs.org/docs

---

**Happy Coding! 🚀**

You now have a complete, professional design system ready to use.
All pages are responsive, all components are reusable, and everything is consistent!

**Status**: ✅ Ready to Build!
