# 🤖 PXEBot - Professional Dashboard Design System

A complete, production-ready design system and UI implementation for PXEBot using **Next.js 14 + TailwindCSS**.

Built with inspiration from **Vercel**, **Linear**, **Notion**, and **Discord** — featuring a clean, premium, modern dark theme that's professional and balanced (not too black, not neon).

---

## 🎨 Design System

### Color Palette
```
Background:     #0B0F19
Surface:        #111827
Border:         #1E293B
Primary:        #3B82F6 (Blue)
Secondary:      #38BDF8 (Cyan)
Success:        #22C55E (Green)
Warning:        #EAB308 (Yellow)
Danger:         #EF4444 (Red)
Text Primary:   #F9FAFB (White)
Text Secondary: #94A3B8 (Gray)
Text Muted:     #64748B (Dark Gray)
```

### Typography
- **Font Family**: Inter, Plus Jakarta Sans, SF Pro Display
- **Title**: 24–28px, font-semibold
- **Subtitle**: 18–20px, font-medium
- **Body**: 14–16px, font-normal
- **Label**: 12px, uppercase, medium, letter-spacing: 0.05em

### Spacing System
- Base units: 8, 12, 16, 24, 32px
- Max container width: 1440px
- 12-column responsive grid

---

## 📦 Project Structure

```
nextjs-redesign/
├── app/
│   ├── page.jsx                 # Landing page
│   ├── layout.jsx               # Root layout
│   ├── globals.css              # Global styles
│   ├── login/page.jsx           # Login page
│   ├── register/page.jsx        # Register page
│   ├── dashboard/page.jsx       # Main dashboard
│   ├── commands/page.jsx        # Commands management
│   ├── timers/page.jsx          # Timers management
│   ├── settings/page.jsx        # Settings page
│   ├── billing/page.jsx         # Billing & plans
│   └── not-found.jsx            # 404 error page
├── components/
│   ├── ui/
│   │   ├── Button.jsx           # Button component
│   │   ├── Card.jsx             # Card component
│   │   ├── Input.jsx            # Input component
│   │   ├── Badge.jsx            # Badge component
│   │   ├── StatCard.jsx         # Stat card component
│   │   └── Toggle.jsx           # Toggle switch
│   └── layout/
│       ├── Header.jsx           # Global header
│       └── Layout.jsx           # Page layout wrapper
├── tailwind.config.js           # Tailwind configuration
├── jsconfig.json                # Path aliases
└── package.json                 # Dependencies
```

---

## 🚀 Getting Started

### Installation

```bash
# Navigate to the project folder
cd nextjs-redesign

# Install dependencies
npm install

# Run development server
npm run dev
```

Open [http://localhost:3000](http://localhost:3000) to view the app.

### Build for Production

```bash
npm run build
npm start
```

---

## 🧩 Components

### Button
```jsx
import Button from '@/components/ui/Button';

<Button variant="primary" size="md" fullWidth>
  Click Me
</Button>
```

**Variants**: `primary`, `secondary`, `danger`, `success`, `ghost`, `link`
**Sizes**: `sm`, `md`, `lg`

---

### Card
```jsx
import Card from '@/components/ui/Card';

<Card
  title="Card Title"
  subtitle="Card subtitle"
  icon={<Icon />}
  hover
>
  Card content here
</Card>
```

---

### Input
```jsx
import Input from '@/components/ui/Input';

<Input
  label="Email"
  type="email"
  placeholder="you@example.com"
  icon={<EmailIcon />}
/>
```

---

### StatCard
```jsx
import StatCard from '@/components/ui/StatCard';

<StatCard
  label="Messages / 1min"
  value="142"
  subtitle="last minute"
  color="primary"
  trend="up"
  trendValue="+12%"
  icon={<ChartIcon />}
/>
```

**Colors**: `primary`, `secondary`, `success`, `warning`, `danger`

---

### Badge
```jsx
import Badge from '@/components/ui/Badge';

<Badge variant="success" size="md">
  Active
</Badge>
```

**Variants**: `default`, `primary`, `success`, `warning`, `danger`, `secondary`

---

### Toggle
```jsx
import Toggle from '@/components/ui/Toggle';

<Toggle
  label="Enable Notifications"
  description="Receive alerts when bot status changes"
  checked={enabled}
  onChange={setEnabled}
/>
```

---

## 📄 Pages

### 1. Landing Page (`/`)
- Hero section with animated background
- Feature cards (6 key features)
- Stats display
- CTA buttons

### 2. Auth Pages (`/login`, `/register`)
- Centered card layout
- Social auth (YouTube)
- Form validation
- Remember me checkbox

### 3. Dashboard (`/dashboard`)
- Stats grid (4 cards)
- Channel card with stream context
- Bot controls (Start/Stop + Roasting toggle)
- Feature buttons (Commands, Timers, Stream Info, Remove)

### 4. Commands (`/commands`)
- Searchable command table
- Filter and export options
- Enable/disable commands
- Usage statistics

### 5. Timers (`/timers`)
- Timer management table
- Interval configuration
- Execution tracking
- Active/paused status

### 6. Settings (`/settings`)
- Profile information
- Password change
- Notification preferences
- Danger zone (account deletion)

### 7. Billing (`/billing`)
- Current plan display
- Plan comparison (Free, Pro, Enterprise)
- Billing history table
- Invoice downloads

### 8. 404 Page (`/not-found`)
- Animated 404 illustration
- Navigation shortcuts
- Help links

---

## 🎯 Design Principles

### Consistency
- All pages use the same color palette
- Uniform spacing and typography
- Consistent component styling
- Same header/navigation across pages

### Responsive
- Mobile-first design
- Breakpoints: `sm: 640px`, `md: 768px`, `lg: 1024px`, `xl: 1280px`
- Touch-friendly (44px minimum tap targets)
- Flexible grids and layouts

### Professional
- Clean, minimal interface
- No excessive gradients or neon colors
- Balanced dark theme
- Smooth transitions and hover effects

### Accessible
- High contrast text
- Clear focus states
- Keyboard navigation support
- Semantic HTML

---

## 🔧 Customization

### Changing Colors

Edit `tailwind.config.js`:

```js
theme: {
  extend: {
    colors: {
      primary: {
        DEFAULT: '#3B82F6', // Change this
      },
    },
  },
}
```

### Adding New Components

1. Create file in `components/ui/`
2. Follow existing component patterns
3. Use Tailwind classes from design system
4. Export and use in pages

### Adding New Pages

1. Create folder in `app/`
2. Add `page.jsx` file
3. Import Layout component
4. Use existing UI components

---

## 📱 Mobile Optimization

All pages are fully responsive:
- Stats cards stack vertically on mobile
- Tables scroll horizontally with custom scrollbar
- Navigation collapses to hamburger menu
- Touch-friendly button sizes
- Proper spacing on all screen sizes

---

## 🎨 Advanced Features

### Animations
- Floating elements with `animate-float`
- Glowing effects with `animate-glow`
- Pulse animations for status badges
- Smooth hover transitions

### Custom Scrollbar
- Styled scrollbar in tables
- Consistent with dark theme
- Hover effects

### Glass Morphism
- Backdrop blur effects
- Semi-transparent surfaces
- Modern, premium look

---

## 🛠️ Tech Stack

- **Framework**: Next.js 14 (App Router)
- **Styling**: TailwindCSS 3.4
- **Font**: Inter (Google Fonts)
- **Icons**: Heroicons (inline SVG)
- **State Management**: React useState hooks

---

## 📋 To-Do (Future Enhancements)

- [ ] Add form validation library (React Hook Form)
- [ ] Implement real API integration
- [ ] Add data fetching (React Query)
- [ ] Add animations library (Framer Motion)
- [ ] Add toast notifications
- [ ] Add modal dialogs
- [ ] Implement dark/light theme toggle
- [ ] Add loading skeletons
- [ ] Add error boundaries

---

## 🤝 Contributing

This is a complete design system. To maintain consistency:
1. Always use the defined color palette
2. Follow the typography scale
3. Maintain spacing system (8/12/16/24/32)
4. Use existing components when possible
5. Test on mobile before committing

---

## 📝 License

Proprietary - PXEBot Dashboard Design System

---

## 🎉 Credits

**Design Inspiration**:
- Vercel Dashboard
- Linear.app
- Notion
- Discord Settings

**Built for**: PXEBot YouTube Chat Bot Platform

---

**Status**: ✅ **Production Ready**
**Version**: 1.0.0
**Last Updated**: 2025-01-09

🚀 **Ready to deploy and scale!**
