'use client';

import React, { useState } from 'react';
import Layout from '@/components/layout/Layout';
import Button from '@/components/ui/Button';
import Card from '@/components/ui/Card';
import Input from '@/components/ui/Input';
import Badge from '@/components/ui/Badge';

const mockUser = {
  name: 'Gaming Channel',
  email: 'user@example.com',
};

const mockCommands = [
  { id: 1, trigger: '!discord', response: 'Join our Discord: discord.gg/example', enabled: true, uses: 142 },
  { id: 2, trigger: '!song', response: 'Current song: {current_song}', enabled: true, uses: 89 },
  { id: 3, trigger: '!socials', response: 'Follow me on Instagram @gaming_pro', enabled: true, uses: 67 },
  { id: 4, trigger: '!schedule', response: 'Stream schedule: Mon-Fri 6PM-12AM', enabled: false, uses: 23 },
];

export default function Commands() {
  const [commands, setCommands] = useState(mockCommands);
  const [searchQuery, setSearchQuery] = useState('');

  const filteredCommands = commands.filter(cmd =>
    cmd.trigger.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <Layout user={mockUser}>
      {/* Page Header */}
      <div className="mb-8">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-2">
          <div>
            <h1 className="text-title text-text-primary">Custom Commands</h1>
            <p className="text-text-secondary mt-1">
              Create and manage automated chat responses
            </p>
          </div>
          <Button variant="primary">
            <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
            </svg>
            <span className="ml-2">Add Command</span>
          </Button>
        </div>
      </div>

      {/* Search and Filters */}
      <Card padding="default" className="mb-6">
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="flex-1">
            <Input
              placeholder="Search commands..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              icon={
                <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                </svg>
              }
            />
          </div>
          <div className="flex gap-2">
            <Button variant="secondary">
              <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 4a1 1 0 011-1h16a1 1 0 011 1v2.586a1 1 0 01-.293.707l-6.414 6.414a1 1 0 00-.293.707V17l-4 4v-6.586a1 1 0 00-.293-.707L3.293 7.293A1 1 0 013 6.586V4z" />
              </svg>
              <span className="ml-2 hidden sm:inline">Filter</span>
            </Button>
            <Button variant="secondary">
              <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4" />
              </svg>
              <span className="ml-2 hidden sm:inline">Export</span>
            </Button>
          </div>
        </div>
      </Card>

      {/* Commands Table */}
      <Card padding="none">
        <div className="overflow-x-auto custom-scrollbar">
          <table className="w-full">
            <thead>
              <tr className="border-b border-border">
                <th className="text-left px-6 py-4 text-label text-text-muted uppercase">Trigger</th>
                <th className="text-left px-6 py-4 text-label text-text-muted uppercase">Response</th>
                <th className="text-left px-6 py-4 text-label text-text-muted uppercase">Uses</th>
                <th className="text-left px-6 py-4 text-label text-text-muted uppercase">Status</th>
                <th className="text-right px-6 py-4 text-label text-text-muted uppercase">Actions</th>
              </tr>
            </thead>
            <tbody>
              {filteredCommands.map((command, index) => (
                <tr
                  key={command.id}
                  className={`hover:bg-hover transition-colors ${
                    index !== filteredCommands.length - 1 ? 'border-b border-border' : ''
                  }`}
                >
                  <td className="px-6 py-4">
                    <code className="px-2 py-1 bg-primary/10 text-primary rounded text-sm font-mono">
                      {command.trigger}
                    </code>
                  </td>
                  <td className="px-6 py-4">
                    <p className="text-sm text-text-secondary max-w-md truncate">
                      {command.response}
                    </p>
                  </td>
                  <td className="px-6 py-4">
                    <span className="text-sm text-text-primary font-medium">
                      {command.uses}
                    </span>
                  </td>
                  <td className="px-6 py-4">
                    <Badge variant={command.enabled ? 'success' : 'default'} size="sm">
                      {command.enabled ? 'Enabled' : 'Disabled'}
                    </Badge>
                  </td>
                  <td className="px-6 py-4">
                    <div className="flex items-center justify-end space-x-2">
                      <button className="p-2 text-text-secondary hover:text-text-primary hover:bg-hover rounded-lg transition-all duration-200">
                        <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
                        </svg>
                      </button>
                      <button className="p-2 text-text-secondary hover:text-danger hover:bg-danger/10 rounded-lg transition-all duration-200">
                        <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                        </svg>
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {/* Empty State */}
        {filteredCommands.length === 0 && (
          <div className="text-center py-12">
            <div className="w-16 h-16 mx-auto mb-4 rounded-full bg-surface flex items-center justify-center">
              <svg className="w-8 h-8 text-text-muted" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 9l3 3-3 3m5 0h3M5 20h14a2 2 0 002-2V6a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
              </svg>
            </div>
            <h3 className="text-lg font-semibold text-text-primary mb-2">No commands found</h3>
            <p className="text-text-secondary mb-4">Try adjusting your search or create a new command</p>
            <Button variant="primary">Add Your First Command</Button>
          </div>
        )}
      </Card>

      {/* Stats Footer */}
      <div className="mt-6 grid grid-cols-3 gap-4">
        <Card padding="default" className="text-center">
          <div className="text-2xl font-bold text-text-primary mb-1">{commands.length}</div>
          <div className="text-sm text-text-secondary">Total Commands</div>
        </Card>
        <Card padding="default" className="text-center">
          <div className="text-2xl font-bold text-text-primary mb-1">
            {commands.filter(c => c.enabled).length}
          </div>
          <div className="text-sm text-text-secondary">Enabled</div>
        </Card>
        <Card padding="default" className="text-center">
          <div className="text-2xl font-bold text-text-primary mb-1">
            {commands.reduce((acc, cmd) => acc + cmd.uses, 0)}
          </div>
          <div className="text-sm text-text-secondary">Total Uses</div>
        </Card>
      </div>
    </Layout>
  );
}
