'use client';

import React, { useState } from 'react';
import Layout from '@/components/layout/Layout';
import Button from '@/components/ui/Button';
import Card from '@/components/ui/Card';
import StatCard from '@/components/ui/StatCard';
import Badge from '@/components/ui/Badge';
import Toggle from '@/components/ui/Toggle';

// Mock user data
const mockUser = {
  name: 'Gaming Channel',
  email: 'user@example.com',
};

export default function Dashboard() {
  const [botRunning, setBotRunning] = useState(false);
  const [roastingMode, setRoastingMode] = useState(false);
  const [loading, setLoading] = useState(false);

  // Mock channel data
  const channel = {
    id: 1,
    name: 'Gaming Pro Channel',
    type: 'Gaming',
    currentGame: 'GTA V',
    status: 'running',
    viewers: 1247,
    streamType: 'gaming',
  };

  const stats = {
    messages1min: 25,
    messages5min: 142,
    uniqueViewers: 1247,
    quotaPerMin: 95,
  };

  const handleToggleBot = async () => {
    setLoading(true);
    setTimeout(() => {
      setBotRunning(!botRunning);
      setLoading(false);
    }, 1000);
  };

  const handleToggleRoasting = async () => {
    const newState = !roastingMode;
    setRoastingMode(newState);
    // API call would go here
  };

  return (
    <Layout user={mockUser}>
      {/* Page Header */}
      <div className="mb-8">
        <div className="flex items-center justify-between mb-2">
          <h1 className="text-title text-text-primary">Dashboard</h1>
          <Button variant="primary" size="sm">
            <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
            </svg>
            <span className="ml-2">Add Channel</span>
          </Button>
        </div>
        <p className="text-text-secondary">
          Monitor your bot's performance and manage your YouTube channels
        </p>
      </div>

      {/* Stats Grid */}
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <StatCard
          label="Messages / 1min"
          value={stats.messages1min}
          subtitle="last minute"
          color="primary"
          trend="up"
          trendValue="+12%"
          icon={
            <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
            </svg>
          }
        />
        <StatCard
          label="Messages / 5min"
          value={stats.messages5min}
          subtitle="last 5 minutes"
          color="secondary"
          icon={
            <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 10h.01M12 10h.01M16 10h.01M9 16H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-5l-5 5v-5z" />
            </svg>
          }
        />
        <StatCard
          label="Unique Viewers"
          value={stats.uniqueViewers.toLocaleString()}
          subtitle="last hour"
          color="success"
          trend="up"
          trendValue="+8%"
          icon={
            <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z" />
            </svg>
          }
        />
        <StatCard
          label="Quota Usage"
          value={`${stats.quotaPerMin}%`}
          subtitle="per minute"
          color="warning"
          icon={
            <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
            </svg>
          }
        />
      </div>

      {/* Channel Card */}
      <Card
        title={channel.name}
        subtitle={`${channel.type} • ${channel.currentGame}`}
        icon={
          <svg className="w-6 h-6" fill="currentColor" viewBox="0 0 24 24">
            <path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z"/>
          </svg>
        }
        action={
          <Badge variant={channel.status === 'running' ? 'success' : 'default'}>
            {channel.status === 'running' ? '● Live' : '○ Stopped'}
          </Badge>
        }
        className="mb-8"
      >
        <div className="space-y-6">
          {/* Stream Context */}
          <div className="p-4 bg-primary/5 border border-primary/20 rounded-xl">
            <div className="flex items-start space-x-3">
              <div className="flex-shrink-0 w-10 h-10 rounded-lg bg-primary/10 flex items-center justify-center text-primary">
                <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
                </svg>
              </div>
              <div className="flex-1">
                <div className="flex items-center space-x-2 mb-1">
                  <span className="text-sm font-medium text-text-primary">Stream Type:</span>
                  <Badge variant="primary" size="sm">{channel.type}</Badge>
                </div>
                <p className="text-sm text-text-secondary">
                  Currently playing <span className="text-primary font-medium">{channel.currentGame}</span>
                </p>
              </div>
            </div>
          </div>

          {/* Bot Controls */}
          <div className="grid sm:grid-cols-2 gap-4">
            <Button
              variant={botRunning ? 'danger' : 'success'}
              fullWidth
              onClick={handleToggleBot}
              loading={loading}
              icon={
                botRunning ? (
                  <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 10h6v4H9z" />
                  </svg>
                ) : (
                  <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z" />
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                  </svg>
                )
              }
            >
              {botRunning ? 'Stop Bot' : 'Start Bot'}
            </Button>

            <div className="flex items-center justify-between p-4 bg-surface border border-border rounded-xl">
              <div className="flex items-center space-x-3">
                <div className={`w-10 h-10 rounded-lg ${roastingMode ? 'bg-danger/10 text-danger' : 'bg-surface'} flex items-center justify-center border ${roastingMode ? 'border-danger/20' : 'border-border'} transition-all duration-200`}>
                  <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17.657 18.657A8 8 0 016.343 7.343S7 9 9 10c0-2 .5-5 2.986-7C14 5 16.09 5.777 17.656 7.343A7.975 7.975 0 0120 13a7.975 7.975 0 01-2.343 5.657z" />
                  </svg>
                </div>
                <div>
                  <p className="text-sm font-medium text-text-primary">Roasting</p>
                  <p className="text-xs text-text-muted">Dark mode</p>
                </div>
              </div>
              <Toggle
                checked={roastingMode}
                onChange={handleToggleRoasting}
              />
            </div>
          </div>

          {/* Feature Buttons */}
          <div className="grid grid-cols-2 lg:grid-cols-4 gap-3">
            <Button variant="secondary" fullWidth className="h-auto py-4 flex-col">
              <svg className="w-6 h-6 mb-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 9l3 3-3 3m5 0h3M5 20h14a2 2 0 002-2V6a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
              </svg>
              <span className="text-sm font-medium">Commands</span>
            </Button>

            <Button variant="secondary" fullWidth className="h-auto py-4 flex-col">
              <svg className="w-6 h-6 mb-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              <span className="text-sm font-medium">Timers</span>
            </Button>

            <Button variant="secondary" fullWidth className="h-auto py-4 flex-col">
              <svg className="w-6 h-6 mb-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              <span className="text-sm font-medium">Stream Info</span>
            </Button>

            <Button variant="ghost" fullWidth className="h-auto py-4 flex-col text-danger hover:text-danger hover:bg-danger/5">
              <svg className="w-6 h-6 mb-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
              </svg>
              <span className="text-sm font-medium">Remove</span>
            </Button>
          </div>
        </div>
      </Card>
    </Layout>
  );
}
