'use client';

import React, { useState } from 'react';
import Layout from '@/components/layout/Layout';
import Button from '@/components/ui/Button';
import Card from '@/components/ui/Card';
import Input from '@/components/ui/Input';
import Badge from '@/components/ui/Badge';

const mockUser = {
  name: 'Gaming Channel',
  email: 'user@example.com',
};

const mockTimers = [
  { id: 1, name: 'Discord Reminder', message: 'Join our Discord: discord.gg/example', interval: 15, enabled: true, executions: 234 },
  { id: 2, name: 'Social Media', message: 'Follow on Instagram @gaming_pro', interval: 20, enabled: true, executions: 189 },
  { id: 3, name: 'Subscribe Reminder', message: 'Like and Subscribe if you\'re enjoying!', interval: 10, enabled: false, executions: 67 },
];

export default function Timers() {
  const [timers, setTimers] = useState(mockTimers);
  const [searchQuery, setSearchQuery] = useState('');

  const filteredTimers = timers.filter(timer =>
    timer.name.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <Layout user={mockUser}>
      {/* Page Header */}
      <div className="mb-8">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-2">
          <div>
            <h1 className="text-title text-text-primary">Automated Timers</h1>
            <p className="text-text-secondary mt-1">
              Schedule recurring messages to keep your chat engaged
            </p>
          </div>
          <Button variant="primary">
            <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
            </svg>
            <span className="ml-2">Add Timer</span>
          </Button>
        </div>
      </div>

      {/* Search and Filters */}
      <Card padding="default" className="mb-6">
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="flex-1">
            <Input
              placeholder="Search timers..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              icon={
                <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                </svg>
              }
            />
          </div>
          <div className="flex gap-2">
            <Button variant="secondary">
              <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 4a1 1 0 011-1h16a1 1 0 011 1v2.586a1 1 0 01-.293.707l-6.414 6.414a1 1 0 00-.293.707V17l-4 4v-6.586a1 1 0 00-.293-.707L3.293 7.293A1 1 0 013 6.586V4z" />
              </svg>
              <span className="ml-2 hidden sm:inline">Filter</span>
            </Button>
          </div>
        </div>
      </Card>

      {/* Timers Table */}
      <Card padding="none">
        <div className="overflow-x-auto custom-scrollbar">
          <table className="w-full">
            <thead>
              <tr className="border-b border-border">
                <th className="text-left px-6 py-4 text-label text-text-muted uppercase">Name</th>
                <th className="text-left px-6 py-4 text-label text-text-muted uppercase">Message</th>
                <th className="text-left px-6 py-4 text-label text-text-muted uppercase">Interval</th>
                <th className="text-left px-6 py-4 text-label text-text-muted uppercase">Executions</th>
                <th className="text-left px-6 py-4 text-label text-text-muted uppercase">Status</th>
                <th className="text-right px-6 py-4 text-label text-text-muted uppercase">Actions</th>
              </tr>
            </thead>
            <tbody>
              {filteredTimers.map((timer, index) => (
                <tr
                  key={timer.id}
                  className={`hover:bg-hover transition-colors ${
                    index !== filteredTimers.length - 1 ? 'border-b border-border' : ''
                  }`}
                >
                  <td className="px-6 py-4">
                    <div className="flex items-center space-x-3">
                      <div className="w-10 h-10 rounded-lg bg-warning/10 flex items-center justify-center text-warning">
                        <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                      </div>
                      <span className="text-sm font-medium text-text-primary">{timer.name}</span>
                    </div>
                  </td>
                  <td className="px-6 py-4">
                    <p className="text-sm text-text-secondary max-w-md truncate">
                      {timer.message}
                    </p>
                  </td>
                  <td className="px-6 py-4">
                    <div className="flex items-center space-x-2">
                      <svg className="w-4 h-4 text-text-muted" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                      </svg>
                      <span className="text-sm text-text-primary font-medium">
                        {timer.interval} min
                      </span>
                    </div>
                  </td>
                  <td className="px-6 py-4">
                    <span className="text-sm text-text-primary font-medium">
                      {timer.executions}
                    </span>
                  </td>
                  <td className="px-6 py-4">
                    <Badge variant={timer.enabled ? 'success' : 'default'} size="sm">
                      {timer.enabled ? 'Active' : 'Paused'}
                    </Badge>
                  </td>
                  <td className="px-6 py-4">
                    <div className="flex items-center justify-end space-x-2">
                      <button className="p-2 text-text-secondary hover:text-text-primary hover:bg-hover rounded-lg transition-all duration-200">
                        <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
                        </svg>
                      </button>
                      <button className="p-2 text-text-secondary hover:text-danger hover:bg-danger/10 rounded-lg transition-all duration-200">
                        <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                        </svg>
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {/* Empty State */}
        {filteredTimers.length === 0 && (
          <div className="text-center py-12">
            <div className="w-16 h-16 mx-auto mb-4 rounded-full bg-surface flex items-center justify-center">
              <svg className="w-8 h-8 text-text-muted" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            </div>
            <h3 className="text-lg font-semibold text-text-primary mb-2">No timers found</h3>
            <p className="text-text-secondary mb-4">Try adjusting your search or create a new timer</p>
            <Button variant="primary">Add Your First Timer</Button>
          </div>
        )}
      </Card>

      {/* Stats Footer */}
      <div className="mt-6 grid grid-cols-3 gap-4">
        <Card padding="default" className="text-center">
          <div className="text-2xl font-bold text-text-primary mb-1">{timers.length}</div>
          <div className="text-sm text-text-secondary">Total Timers</div>
        </Card>
        <Card padding="default" className="text-center">
          <div className="text-2xl font-bold text-text-primary mb-1">
            {timers.filter(t => t.enabled).length}
          </div>
          <div className="text-sm text-text-secondary">Active</div>
        </Card>
        <Card padding="default" className="text-center">
          <div className="text-2xl font-bold text-text-primary mb-1">
            {timers.reduce((acc, timer) => acc + timer.executions, 0)}
          </div>
          <div className="text-sm text-text-secondary">Total Executions</div>
        </Card>
      </div>
    </Layout>
  );
}
