<?php
/**
 * Manual Polling Test
 * Manually trigger one polling cycle
 */

set_time_limit(120);
ini_set('display_errors', 1);
error_reporting(E_ALL);

require __DIR__.'/../vendor/autoload.php';
$app = require_once __DIR__.'/../bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="en" class="dark">
<head>
    <meta charset="UTF-8">
    <title>Manual Poll - PHXBot</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-900 text-gray-100 p-8">
    <div class="max-w-4xl mx-auto">
        <div class="bg-gray-800 rounded-lg shadow-2xl p-8 border border-gray-700">
            <h1 class="text-3xl font-bold text-purple-400 mb-6">🔄 Manual Poll Test</h1>
            <p class="text-gray-400 mb-6">This will manually poll YouTube chat once and process any messages.</p>

            <?php
            // Get tenant ID from URL parameter or use first running tenant with live_chat_id
            $tenantId = $_GET['tenant_id'] ?? null;

            if ($tenantId) {
                $tenant = \App\Models\Tenant::where('id', $tenantId)->where('status', 'running')->first();
            } else {
                // Prefer tenant with live_chat_id (active stream)
                $tenant = \App\Models\Tenant::where('status', 'running')
                    ->whereNotNull('live_chat_id')
                    ->first();

                // If no tenant with live_chat_id, get any running tenant
                if (!$tenant) {
                    $tenant = \App\Models\Tenant::where('status', 'running')->first();
                }
            }

            if (!$tenant) {
                echo "<p class='text-red-400'>❌ No running tenant. Start bot from dashboard.</p>";
                echo "</body></html>";
                exit;
            }

            // Show all running tenants for selection
            $allRunningTenants = \App\Models\Tenant::where('status', 'running')->get();

            if ($allRunningTenants->count() > 1) {
                echo "<div class='mb-4 bg-purple-900/20 border border-purple-500/30 rounded-lg p-4'>";
                echo "<p class='text-sm text-gray-300 mb-3'>Multiple channels running. Select one:</p>";
                echo "<div class='flex gap-2 flex-wrap'>";
                foreach ($allRunningTenants as $t) {
                    $isActive = $t->id === $tenant->id;
                    $hasLiveChat = !empty($t->live_chat_id);
                    $buttonClass = $isActive ? 'bg-purple-600 text-white' : 'bg-gray-700 text-gray-300 hover:bg-gray-600';
                    $liveBadge = $hasLiveChat ? ' 🔴' : '';
                    echo "<a href='?tenant_id={$t->id}' class='px-4 py-2 rounded {$buttonClass}'>{$t->channel_title}{$liveBadge}</a>";
                }
                echo "</div>";
                echo "</div>";
            }

            echo "<div class='mb-4 bg-gray-900 p-3 rounded'>";
            echo "<p class='text-gray-300'>Channel: <strong>{$tenant->channel_title}</strong></p>";
            echo "<p class='text-gray-300'>Bot Status: <strong class='text-green-400'>RUNNING</strong></p>";
            if ($tenant->live_chat_id) {
                echo "<p class='text-gray-300'>Live Stream: <strong class='text-green-400'>🔴 ACTIVE</strong></p>";
            } else {
                echo "<p class='text-gray-300'>Live Stream: <strong class='text-yellow-400'>⚠️ NO ACTIVE STREAM</strong></p>";
            }
            echo "</div>";

            // Step 1: Get/Set Live Chat ID
            echo "<div class='mb-4 bg-gray-900 p-3 rounded border border-gray-700'>";
            echo "<h2 class='text-lg font-bold text-indigo-400'>Step 1: Get Live Chat ID</h2>";

            $youtubeService = app(\App\Services\YouTubeService::class);
            $broadcast = $youtubeService->getActiveBroadcast($tenant);

            if (!$broadcast || !$broadcast['live_chat_id']) {
                echo "<p class='text-red-400 mt-2'>❌ No active stream found!</p>";
                echo "<p class='text-yellow-300 text-sm mt-1'>Start a live stream first.</p>";
                echo "</div></body></html>";
                exit;
            }

            echo "<p class='text-green-400 mt-2'>✅ Active stream found!</p>";
            echo "<p class='text-gray-400 text-sm'>Live Chat ID: <code class='text-xs'>{$broadcast['live_chat_id']}</code></p>";

            // Update tenant with live_chat_id
            $tenant->update([
                'live_chat_id' => $broadcast['live_chat_id'],
                'broadcast_id' => $broadcast['broadcast_id'],
            ]);

            echo "<p class='text-green-400 text-sm mt-1'>✅ Saved to database</p>";
            echo "</div>";

            // Step 2: Poll Messages
            echo "<div class='mb-4 bg-gray-900 p-3 rounded border border-gray-700'>";
            echo "<h2 class='text-lg font-bold text-indigo-400'>Step 2: Poll Chat Messages</h2>";

            try {
                $result = $youtubeService->pollMessages($tenant, $tenant->next_page_token);

                if (!$result) {
                    echo "<p class='text-red-400 mt-2'>❌ Polling failed!</p>";
                    echo "<p class='text-gray-400 text-sm'>Check Laravel logs for details.</p>";
                } else {
                    $messageCount = count($result['messages']);
                    echo "<p class='text-green-400 mt-2'>✅ Polled successfully!</p>";
                    echo "<p class='text-gray-400 text-sm'>Messages fetched: <strong>$messageCount</strong></p>";
                    echo "<p class='text-gray-400 text-sm'>Next page token: " . ($result['next_page_token'] ?? 'null') . "</p>";

                    // Update next page token
                    $tenant->update([
                        'next_page_token' => $result['next_page_token'],
                        'last_polled_at' => now(),
                    ]);

                    // Step 3: Process Messages
                    if ($messageCount > 0) {
                        echo "</div>";
                        echo "<div class='mb-4 bg-gray-900 p-3 rounded border border-gray-700'>";
                        echo "<h2 class='text-lg font-bold text-indigo-400'>Step 3: Process Messages</h2>";

                        $processed = 0;
                        $queued = 0;

                        foreach ($result['messages'] as $msg) {
                            $messageId = $msg['id'];
                            $authorChannelId = $msg['authorDetails']['channelId'] ?? 'unknown';
                            $authorName = $msg['authorDetails']['displayName'] ?? 'Unknown';
                            $messageText = $msg['snippet']['displayMessage'] ?? '';
                            $publishedAt = $msg['snippet']['publishedAt'] ?? null;

                            echo "<div class='mb-2 pb-2 border-b border-gray-700'>";
                            echo "<p class='text-xs text-gray-500'>ID: $messageId</p>";
                            echo "<p class='text-sm'><strong class='text-indigo-300'>$authorName:</strong> <span class='text-gray-300'>$messageText</span></p>";

                            // Check if already processed
                            $exists = \DB::table('chat_messages')
                                ->where('message_id', $messageId)
                                ->exists();

                            if ($exists) {
                                echo "<p class='text-yellow-400 text-xs mt-1'>⚠️ Already processed (skipped)</p>";
                                $processed++;
                            } else {
                                // Save to database
                                \DB::table('chat_messages')->insert([
                                    'tenant_id' => $tenant->id,
                                    'message_id' => $messageId,
                                    'author_channel_id' => $authorChannelId,
                                    'author_name' => $authorName,
                                    'message_text' => $messageText,
                                    'published_at' => $publishedAt ? \Carbon\Carbon::parse($publishedAt)->format('Y-m-d H:i:s') : null,
                                    'processed' => false,
                                    'created_at' => now(),
                                    'updated_at' => now(),
                                ]);

                                // Queue job
                                \App\Jobs\ProcessIncomingMessage::dispatch($tenant, [
                                    'id' => $messageId,
                                    'snippet' => $msg['snippet'],
                                    'authorDetails' => $msg['authorDetails'],
                                ]);

                                echo "<p class='text-green-400 text-xs mt-1'>✅ Saved & queued for processing</p>";
                                $queued++;
                            }

                            echo "</div>";
                        }

                        echo "<p class='text-green-400 mt-3'>✅ Processed: $processed, Queued: $queued</p>";
                    }
                }

            } catch (\Exception $e) {
                echo "<p class='text-red-400 mt-2'>❌ Error: " . htmlspecialchars($e->getMessage()) . "</p>";
            }

            echo "</div>";

            // Step 4: Process Queue
            echo "<div class='mb-4 bg-gray-900 p-3 rounded border border-gray-700'>";
            echo "<h2 class='text-lg font-bold text-indigo-400'>Step 4: Process Queue</h2>";

            try {
                $jobCount = \DB::table('jobs')->count();
                echo "<p class='text-gray-300 mt-2'>Jobs in queue: <strong>$jobCount</strong></p>";

                if ($jobCount > 0) {
                    echo "<p class='text-yellow-300 text-sm mt-1'>Processing jobs...</p>";

                    // Manually process jobs
                    \Artisan::call('queue:work', [
                        'connection' => 'database',
                        '--once' => true,
                        '--tries' => 1,
                    ]);

                    $output = \Artisan::output();
                    echo "<pre class='bg-gray-800 p-2 rounded text-xs mt-2 text-green-300'>" . htmlspecialchars($output) . "</pre>";

                    $jobCountAfter = \DB::table('jobs')->count();
                    echo "<p class='text-gray-300 mt-2'>Jobs remaining: <strong>$jobCountAfter</strong></p>";

                    if ($jobCountAfter < $jobCount) {
                        echo "<p class='text-green-400 text-sm'>✅ Job processed successfully!</p>";
                        echo "<p class='text-yellow-300 text-sm mt-2'>Check your stream chat - bot should have replied!</p>";
                    }
                } else {
                    echo "<p class='text-yellow-400 text-sm mt-1'>No jobs to process</p>";
                }

            } catch (\Exception $e) {
                echo "<p class='text-red-400 mt-2'>❌ Error: " . htmlspecialchars($e->getMessage()) . "</p>";
            }

            echo "</div>";

            // Summary
            echo "<div class='bg-green-900/20 border border-green-500/50 rounded p-4'>";
            echo "<p class='text-green-300 font-semibold'>✅ Manual Poll Complete!</p>";
            echo "<p class='text-gray-400 text-sm mt-2'>Check your stream chat to see if bot replied.</p>";
            echo "<p class='text-yellow-300 text-sm mt-3'>For automatic replies, make sure cron jobs are running!</p>";
            echo "</div>";
            ?>

            <div class="mt-6 text-center space-x-3">
                <a href="?run=1" class="px-4 py-2 bg-indigo-600 rounded">🔄 Poll Again</a>
                <a href="/check-system.php" class="px-4 py-2 bg-gray-700 rounded">Check System</a>
            </div>
        </div>
    </div>
</body>
</html>
