<?php
/**
 * Ultra-Fast Parallel Chunked Upload
 * Multiple chunks upload simultaneously for maximum speed
 */

// Maximum performance settings
ini_set('upload_max_filesize', '50G');
ini_set('post_max_size', '50G');
ini_set('max_execution_time', '0');
ini_set('max_input_time', '0');
ini_set('memory_limit', '-1');
set_time_limit(0);

// Disable buffering for immediate response
if (ob_get_level()) ob_end_clean();

// Upload directory
$uploadDir = __DIR__ . '/uploads/';

// Create uploads directory
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Handle chunked upload
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // Chunk upload handling
    if (isset($_POST['chunk']) && isset($_POST['chunks'])) {
        $chunk = intval($_POST['chunk']);
        $chunks = intval($_POST['chunks']);
        $fileName = $_POST['fileName'];
        $uniqueId = $_POST['uniqueId'];

        // Sanitize filename
        $fileName = preg_replace('/[^a-zA-Z0-9._-]/', '_', $fileName);

        // Temporary chunk directory
        $tempDir = $uploadDir . 'temp_' . $uniqueId . '/';
        if (!is_dir($tempDir)) {
            mkdir($tempDir, 0755, true);
        }

        // Save chunk to temp directory
        $chunkFile = $tempDir . 'chunk_' . str_pad($chunk, 6, '0', STR_PAD_LEFT);
        $chunkData = file_get_contents($_FILES['file']['tmp_name']);
        file_put_contents($chunkFile, $chunkData);

        // Check if all chunks are uploaded
        $uploadedChunks = glob($tempDir . 'chunk_*');

        if (count($uploadedChunks) == $chunks) {
            // All chunks uploaded - merge them
            $finalFile = $uploadDir . $uniqueId . '_' . $fileName;
            $out = fopen($finalFile, 'wb');

            // Sort chunks by name
            sort($uploadedChunks);

            // Merge all chunks
            foreach ($uploadedChunks as $chunkPath) {
                $chunkContent = file_get_contents($chunkPath);
                fwrite($out, $chunkContent);
                unlink($chunkPath); // Delete chunk after merging
            }
            fclose($out);

            // Remove temp directory
            rmdir($tempDir);

            $fileSize = filesize($finalFile);
            $fileSizeMB = round($fileSize / (1024 * 1024), 2);

            echo json_encode([
                'success' => true,
                'complete' => true,
                'message' => 'File uploaded successfully!',
                'filename' => $uniqueId . '_' . $fileName,
                'original_name' => $fileName,
                'size' => $fileSizeMB . ' MB',
                'path' => '/uploads/' . $uniqueId . '_' . $fileName
            ]);
        } else {
            // Chunk uploaded successfully, waiting for more
            echo json_encode([
                'success' => true,
                'complete' => false,
                'chunk' => $chunk,
                'uploaded' => count($uploadedChunks),
                'total' => $chunks
            ]);
        }
        exit;
    }

    // Standard upload fallback
    if (isset($_FILES['file'])) {
        $file = $_FILES['file'];

        if ($file['error'] !== UPLOAD_ERR_OK) {
            echo json_encode(['success' => false, 'message' => 'Upload error: ' . $file['error']]);
            exit;
        }

        $originalName = basename($file['name']);
        $extension = pathinfo($originalName, PATHINFO_EXTENSION);
        $filename = time() . '_' . uniqid() . '.' . $extension;
        $destination = $uploadDir . $filename;

        if (move_uploaded_file($file['tmp_name'], $destination)) {
            $fileSize = filesize($destination);
            $fileSizeMB = round($fileSize / (1024 * 1024), 2);

            echo json_encode([
                'success' => true,
                'complete' => true,
                'message' => 'File uploaded successfully!',
                'filename' => $filename,
                'original_name' => $originalName,
                'size' => $fileSizeMB . ' MB',
                'path' => '/uploads/' . $filename
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to move uploaded file']);
        }
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>⚡ Ultra-Fast Upload</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            padding: 40px;
            max-width: 600px;
            width: 100%;
        }

        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 28px;
        }

        .subtitle {
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }

        .speed-badge {
            display: inline-block;
            background: linear-gradient(135deg, #f59e0b 0%, #ef4444 100%);
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            margin-left: 10px;
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }

        .upload-area {
            border: 3px dashed #667eea;
            border-radius: 15px;
            padding: 60px 20px;
            text-align: center;
            background: #f8f9ff;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-bottom: 20px;
        }

        .upload-area:hover {
            border-color: #764ba2;
            background: #f0f1ff;
            transform: scale(1.02);
        }

        .upload-area.dragover {
            border-color: #22c55e;
            background: #f0fdf4;
            transform: scale(1.05);
        }

        .upload-icon {
            font-size: 64px;
            margin-bottom: 15px;
        }

        .upload-text {
            color: #667eea;
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 8px;
        }

        .upload-hint {
            color: #999;
            font-size: 14px;
        }

        input[type="file"] {
            display: none;
        }

        .file-info {
            background: #f8f9ff;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
            display: none;
        }

        .file-info.active {
            display: block;
        }

        .file-name {
            color: #333;
            font-weight: 600;
            margin-bottom: 5px;
            word-break: break-all;
        }

        .file-size {
            color: #666;
            font-size: 14px;
        }

        .progress-container {
            background: #e5e7eb;
            border-radius: 10px;
            height: 50px;
            overflow: hidden;
            margin-bottom: 15px;
            display: none;
            position: relative;
        }

        .progress-container.active {
            display: block;
        }

        .progress-bar {
            background: linear-gradient(90deg, #f59e0b 0%, #ef4444 100%);
            height: 100%;
            width: 0%;
            transition: width 0.1s ease;
            position: relative;
            box-shadow: 0 0 20px rgba(245, 158, 11, 0.5);
        }

        .progress-text {
            position: absolute;
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 700;
            font-size: 16px;
            z-index: 1;
            text-shadow: 0 2px 4px rgba(0,0,0,0.3);
        }

        .upload-stats {
            display: none;
            grid-template-columns: repeat(4, 1fr);
            gap: 10px;
            margin-bottom: 20px;
        }

        .upload-stats.active {
            display: grid;
        }

        .stat-item {
            background: #f8f9ff;
            border-radius: 8px;
            padding: 10px;
            text-align: center;
        }

        .stat-label {
            color: #999;
            font-size: 10px;
            margin-bottom: 4px;
        }

        .stat-value {
            color: #333;
            font-weight: 700;
            font-size: 14px;
        }

        .upload-btn {
            background: linear-gradient(135deg, #f59e0b 0%, #ef4444 100%);
            color: white;
            border: none;
            padding: 15px 40px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            width: 100%;
            transition: all 0.2s ease;
            display: none;
            box-shadow: 0 4px 15px rgba(245, 158, 11, 0.4);
        }

        .upload-btn.active {
            display: block;
        }

        .upload-btn:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(245, 158, 11, 0.6);
        }

        .upload-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        .message {
            padding: 15px;
            border-radius: 10px;
            margin-top: 20px;
            display: none;
            font-weight: 500;
            animation: slideIn 0.3s ease;
        }

        .message.active {
            display: block;
        }

        .message.success {
            background: #d1fae5;
            color: #065f46;
            border: 2px solid #10b981;
        }

        .message.error {
            background: #fee2e2;
            color: #991b1b;
            border: 2px solid #ef4444;
        }

        .uploaded-files {
            margin-top: 30px;
        }

        .uploaded-file-item {
            background: #f8f9ff;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            animation: slideIn 0.3s ease;
        }

        .uploaded-file-info {
            flex: 1;
        }

        .uploaded-file-name {
            color: #333;
            font-weight: 600;
            margin-bottom: 5px;
        }

        .uploaded-file-size {
            color: #666;
            font-size: 12px;
        }

        .download-btn {
            background: #667eea;
            color: white;
            border: none;
            padding: 8px 20px;
            border-radius: 6px;
            font-size: 14px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
            transition: all 0.2s ease;
        }

        .download-btn:hover {
            background: #764ba2;
            transform: scale(1.05);
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .spinner {
            display: inline-block;
            animation: spin 1s linear infinite;
        }

        .speed-indicator {
            color: #ef4444;
            font-weight: 700;
        }

        .parallel-indicator {
            background: #fef3c7;
            border: 2px solid #f59e0b;
            border-radius: 8px;
            padding: 10px;
            margin-bottom: 20px;
            text-align: center;
            display: none;
        }

        .parallel-indicator.active {
            display: block;
        }

        .parallel-text {
            color: #92400e;
            font-size: 13px;
            font-weight: 600;
        }

        .parallel-count {
            color: #ef4444;
            font-weight: 700;
            font-size: 18px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>⚡ Ultra-Fast Upload <span class="speed-badge">PARALLEL MODE</span></h1>
        <p class="subtitle">Maximum speed with simultaneous multi-chunk uploads</p>

        <!-- Upload Area -->
        <div class="upload-area" id="uploadArea">
            <div class="upload-icon">🔥</div>
            <div class="upload-text">Click to select file or drag & drop</div>
            <div class="upload-hint">Unlimited size • Ultra-fast parallel upload • All types</div>
        </div>

        <input type="file" id="fileInput" accept="*/*">

        <!-- File Info -->
        <div class="file-info" id="fileInfo">
            <div class="file-name" id="fileName"></div>
            <div class="file-size" id="fileSize"></div>
        </div>

        <!-- Parallel Upload Indicator -->
        <div class="parallel-indicator" id="parallelIndicator">
            <div class="parallel-text">Uploading <span class="parallel-count" id="parallelCount">0</span> chunks simultaneously</div>
        </div>

        <!-- Upload Stats -->
        <div class="upload-stats" id="uploadStats">
            <div class="stat-item">
                <div class="stat-label">Speed</div>
                <div class="stat-value speed-indicator" id="uploadSpeed">0 MB/s</div>
            </div>
            <div class="stat-item">
                <div class="stat-label">Uploaded</div>
                <div class="stat-value" id="uploadedSize">0 MB</div>
            </div>
            <div class="stat-item">
                <div class="stat-label">Time Left</div>
                <div class="stat-value" id="timeLeft">--</div>
            </div>
            <div class="stat-item">
                <div class="stat-label">Chunks</div>
                <div class="stat-value" id="chunkProgress">0/0</div>
            </div>
        </div>

        <!-- Progress Bar -->
        <div class="progress-container" id="progressContainer">
            <div class="progress-bar" id="progressBar">
                <div class="progress-text" id="progressText">0%</div>
            </div>
        </div>

        <!-- Upload Button -->
        <button class="upload-btn" id="uploadBtn">
            🔥 Start Ultra-Fast Upload
        </button>

        <!-- Message -->
        <div class="message" id="message"></div>

        <!-- Uploaded Files List -->
        <div class="uploaded-files" id="uploadedFiles"></div>
    </div>

    <script>
        const uploadArea = document.getElementById('uploadArea');
        const fileInput = document.getElementById('fileInput');
        const fileInfo = document.getElementById('fileInfo');
        const fileName = document.getElementById('fileName');
        const fileSize = document.getElementById('fileSize');
        const uploadBtn = document.getElementById('uploadBtn');
        const progressContainer = document.getElementById('progressContainer');
        const progressBar = document.getElementById('progressBar');
        const progressText = document.getElementById('progressText');
        const message = document.getElementById('message');
        const uploadedFiles = document.getElementById('uploadedFiles');
        const uploadStats = document.getElementById('uploadStats');
        const uploadSpeed = document.getElementById('uploadSpeed');
        const uploadedSize = document.getElementById('uploadedSize');
        const timeLeft = document.getElementById('timeLeft');
        const chunkProgress = document.getElementById('chunkProgress');
        const parallelIndicator = document.getElementById('parallelIndicator');
        const parallelCount = document.getElementById('parallelCount');

        let selectedFile = null;
        const CHUNK_SIZE = 2 * 1024 * 1024; // 2MB chunks for faster parallel upload
        const MAX_PARALLEL = 6; // Upload 6 chunks simultaneously

        // Click to select file
        uploadArea.addEventListener('click', () => {
            fileInput.click();
        });

        // Drag & Drop
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('dragover');
        });

        uploadArea.addEventListener('dragleave', () => {
            uploadArea.classList.remove('dragover');
        });

        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');

            if (e.dataTransfer.files.length > 0) {
                handleFileSelect(e.dataTransfer.files[0]);
            }
        });

        // File input change
        fileInput.addEventListener('change', (e) => {
            if (e.target.files.length > 0) {
                handleFileSelect(e.target.files[0]);
            }
        });

        // Handle file selection
        function handleFileSelect(file) {
            selectedFile = file;

            fileName.textContent = file.name;
            fileSize.textContent = formatFileSize(file.size);
            fileInfo.classList.add('active');
            uploadBtn.classList.add('active');
            message.classList.remove('active');
        }

        // Format file size
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
        }

        // Upload file with parallel chunking
        uploadBtn.addEventListener('click', async () => {
            if (!selectedFile) {
                showMessage('Please select a file first', 'error');
                return;
            }

            const totalSize = selectedFile.size;
            const chunks = Math.ceil(totalSize / CHUNK_SIZE);
            const uniqueId = Date.now() + '_' + Math.random().toString(36).substr(2, 9);

            let uploadedChunks = 0;
            const startTime = Date.now();

            uploadBtn.disabled = true;
            uploadBtn.innerHTML = '<span class="spinner">⏳</span> Uploading...';
            progressContainer.classList.add('active');
            uploadStats.classList.add('active');
            parallelIndicator.classList.add('active');

            // Upload chunks in parallel batches
            for (let i = 0; i < chunks; i += MAX_PARALLEL) {
                const batchPromises = [];
                const batchSize = Math.min(MAX_PARALLEL, chunks - i);

                parallelCount.textContent = batchSize;

                for (let j = 0; j < batchSize; j++) {
                    const chunkIndex = i + j;
                    if (chunkIndex >= chunks) break;

                    const start = chunkIndex * CHUNK_SIZE;
                    const end = Math.min(start + CHUNK_SIZE, totalSize);
                    const chunk = selectedFile.slice(start, end);

                    const formData = new FormData();
                    formData.append('file', chunk);
                    formData.append('chunk', chunkIndex);
                    formData.append('chunks', chunks);
                    formData.append('fileName', selectedFile.name);
                    formData.append('uniqueId', uniqueId);

                    // Add chunk upload promise to batch
                    batchPromises.push(
                        fetch('upload.php', {
                            method: 'POST',
                            body: formData
                        })
                        .then(response => response.json())
                        .then(result => {
                            if (!result.success) {
                                throw new Error(result.message || 'Upload failed');
                            }
                            return result;
                        })
                    );
                }

                try {
                    // Wait for all chunks in this batch to complete
                    const results = await Promise.all(batchPromises);

                    uploadedChunks += batchSize;

                    // Update progress
                    const uploadedBytes = Math.min(uploadedChunks * CHUNK_SIZE, totalSize);
                    const percentComplete = Math.round((uploadedChunks / chunks) * 100);
                    progressBar.style.width = percentComplete + '%';
                    progressText.textContent = percentComplete + '%';

                    // Calculate upload speed
                    const elapsedSeconds = (Date.now() - startTime) / 1000;
                    const speedMBps = (uploadedBytes / (1024 * 1024)) / elapsedSeconds;
                    uploadSpeed.textContent = speedMBps.toFixed(2) + ' MB/s';

                    // Calculate uploaded size
                    uploadedSize.textContent = formatFileSize(uploadedBytes);

                    // Calculate time left
                    const remainingChunks = chunks - uploadedChunks;
                    const remainingSeconds = (remainingChunks / batchSize) * (elapsedSeconds / (i / MAX_PARALLEL + 1));
                    timeLeft.textContent = remainingSeconds > 0 ? Math.ceil(remainingSeconds) + 's' : '0s';

                    // Update chunk progress
                    chunkProgress.textContent = uploadedChunks + '/' + chunks;

                    // Check if upload is complete
                    const lastResult = results[results.length - 1];
                    if (lastResult.complete) {
                        showMessage(lastResult.message, 'success');
                        addUploadedFile(lastResult);
                        resetForm();
                        return;
                    }

                } catch (error) {
                    showMessage('Upload failed: ' + error.message, 'error');
                    resetForm();
                    return;
                }
            }
        });

        // Show message
        function showMessage(text, type) {
            message.textContent = text;
            message.className = 'message active ' + type;
        }

        // Add uploaded file to list
        function addUploadedFile(fileData) {
            const fileItem = document.createElement('div');
            fileItem.className = 'uploaded-file-item';
            fileItem.innerHTML = `
                <div class="uploaded-file-info">
                    <div class="uploaded-file-name">✅ ${fileData.original_name}</div>
                    <div class="uploaded-file-size">${fileData.size}</div>
                </div>
                <a href="${fileData.path}" class="download-btn" download>Download</a>
            `;
            uploadedFiles.insertBefore(fileItem, uploadedFiles.firstChild);
        }

        // Reset form
        function resetForm() {
            selectedFile = null;
            fileInput.value = '';
            fileInfo.classList.remove('active');
            uploadBtn.classList.remove('active');
            uploadBtn.disabled = false;
            uploadBtn.innerHTML = '🔥 Start Ultra-Fast Upload';
            progressBar.style.width = '0%';
            progressText.textContent = '0%';
            progressContainer.classList.remove('active');
            uploadStats.classList.remove('active');
            parallelIndicator.classList.remove('active');
        }
    </script>
</body>
</html>
