@extends('layouts.app')

@section('content')
<!-- Main Dashboard Container -->
<div class="min-h-[calc(100vh-4rem)] bg-body">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6 lg:py-10">

        <!-- Empty State -->
        @if($tenants->isEmpty())
        <div class="flex items-center justify-center min-h-[60vh]">
            <div class="text-center max-w-lg px-4">
                <!-- Empty State Icon -->
                <div class="relative mx-auto w-24 h-24 mb-8">
                    <div class="absolute inset-0 bg-gradient-to-br from-accent-primary/20 to-accent-secondary/20 rounded-3xl blur-xl"></div>
                    <div class="relative w-24 h-24 bg-card border border-subtle rounded-3xl flex items-center justify-center shadow-2xl">
                        <svg class="w-12 h-12 text-accent-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z" />
                        </svg>
                    </div>
                </div>
                <h2 class="text-3xl font-bold text-main mb-4">Welcome to DAYBot</h2>
                <p class="text-base text-muted mb-10 leading-relaxed max-w-md mx-auto">
                    Connect your YouTube channel to start managing your AI-powered chatbot and engage with your live stream audience.
                </p>
                <a href="{{ route('oauth.google.redirect') }}"
                   class="inline-flex items-center px-8 py-4 bg-white hover:bg-gray-50 text-gray-800 text-base font-semibold rounded-xl border border-gray-200 shadow-lg hover:shadow-xl transition-all duration-200 hover:-translate-y-0.5">
                    <svg class="w-6 h-6 mr-3" viewBox="0 0 24 24">
                        <path fill="#4285F4" d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z"/>
                        <path fill="#34A853" d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z"/>
                        <path fill="#FBBC05" d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z"/>
                        <path fill="#EA4335" d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z"/>
                    </svg>
                    Connect with Google
                </a>
            </div>
        </div>
        @endif

        <!-- Channels Loop -->
        @foreach($dashboardData as $data)
        @php
            $tenant = $data['tenant'];
            $stats = $data['stats'];
            $quota = $data['quota'];
            $roastingMode = $data['roasting_mode'];
        @endphp

        <div class="space-y-6 lg:space-y-8 mb-12 last:mb-0">
            <!-- =============================================== -->
            <!-- SECTION 1: Channel Header & Primary Controls -->
            <!-- =============================================== -->
            <section class="bg-card border border-subtle rounded-2xl overflow-hidden shadow-xl">
                <!-- Channel Info Bar -->
                <div class="px-6 py-5 border-b border-subtle bg-gradient-to-r from-accent-primary/5 to-accent-secondary/5">
                    <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
                        <!-- Left: Channel Name & Status -->
                        <div class="flex items-center gap-4">
                            <div class="flex-shrink-0 w-12 h-12 rounded-xl bg-gradient-to-br from-accent-primary to-accent-secondary flex items-center justify-center shadow-lg">
                                <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z" />
                                </svg>
                            </div>
                            <div>
                                <div class="flex items-center gap-3 flex-wrap">
                                    <h1 class="text-xl lg:text-2xl font-bold text-main">{{ $tenant->channel_title }}</h1>
                                    <!-- Status Badge -->
                                    @if($tenant->status === 'running')
                                    <span class="inline-flex items-center gap-1.5 px-3 py-1 rounded-full text-xs font-semibold bg-success/15 text-success border border-success/25">
                                        <span class="w-2 h-2 rounded-full bg-success animate-pulse"></span>
                                        RUNNING
                                    </span>
                                    @elseif($tenant->status === 'error')
                                    <span class="inline-flex items-center gap-1.5 px-3 py-1 rounded-full text-xs font-semibold bg-danger/15 text-danger border border-danger/25">
                                        <span class="w-2 h-2 rounded-full bg-danger"></span>
                                        ERROR
                                    </span>
                                    @else
                                    <span class="inline-flex items-center gap-1.5 px-3 py-1 rounded-full text-xs font-semibold bg-surface text-muted border border-subtle">
                                        <span class="w-2 h-2 rounded-full bg-gray-500"></span>
                                        STOPPED
                                    </span>
                                    @endif
                                </div>
                                <p class="text-sm text-muted mt-1">
                                    <span class="font-mono text-xs bg-surface px-2 py-0.5 rounded">{{ $tenant->channel_id }}</span>
                                </p>
                            </div>
                        </div>

                        <!-- Right: Add Account Button -->
                        <a href="{{ route('oauth.google.redirect') }}"
                           class="inline-flex items-center justify-center px-4 py-2.5 bg-white hover:bg-gray-50 text-gray-700 text-sm font-medium rounded-lg border border-gray-200 shadow-sm hover:shadow transition-all">
                            <svg class="w-4 h-4 mr-2" viewBox="0 0 24 24">
                                <path fill="#4285F4" d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z"/>
                                <path fill="#34A853" d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z"/>
                                <path fill="#FBBC05" d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z"/>
                                <path fill="#EA4335" d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z"/>
                            </svg>
                            Add Account
                        </a>
                    </div>
                </div>

                <!-- Primary Controls Row -->
                <div class="px-6 py-5">
                    <div class="flex flex-col sm:flex-row gap-4">
                        <!-- Start/Stop Bot Button -->
                        @if($tenant->status === 'running')
                        <form method="POST" action="{{ route('tenant.stop', $tenant) }}" class="flex-1 sm:flex-initial">
                            @csrf
                            <button type="submit"
                                    onclick="return confirm('Are you sure you want to stop the bot?')"
                                    class="w-full sm:w-auto inline-flex items-center justify-center gap-2 px-8 py-3.5 bg-danger hover:bg-red-600 text-white text-sm font-semibold rounded-xl shadow-lg shadow-danger/25 hover:shadow-xl hover:shadow-danger/30 transition-all duration-200 hover:-translate-y-0.5">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                    <rect x="9" y="9" width="6" height="6" rx="1" fill="currentColor"/>
                                </svg>
                                Stop Bot
                            </button>
                        </form>
                        @else
                        <form method="POST" action="{{ route('tenant.start', $tenant) }}" class="flex-1 sm:flex-initial">
                            @csrf
                            <button type="submit"
                                    class="w-full sm:w-auto inline-flex items-center justify-center gap-2 px-8 py-3.5 bg-success hover:bg-green-600 text-white text-sm font-semibold rounded-xl shadow-lg shadow-success/25 hover:shadow-xl hover:shadow-success/30 transition-all duration-200 hover:-translate-y-0.5">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z" />
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                                Start Bot
                            </button>
                        </form>
                        @endif

                        <!-- Roasting Mode Toggle -->
                        <div class="flex items-center gap-4 px-5 py-3 bg-surface/50 border border-subtle rounded-xl" id="roasting-container-{{ $tenant->id }}">
                            <div class="flex items-center gap-3 flex-1">
                                <div class="w-10 h-10 rounded-lg flex items-center justify-center transition-all {{ $roastingMode ? 'bg-orange-500/15 text-orange-400' : 'bg-surface text-muted' }}" id="roasting-icon-{{ $tenant->id }}">
                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 18.657A8 8 0 016.343 7.343S7 9 9 10c0-2 .5-5 2.986-7C14 5 16.09 5.777 17.656 7.343A7.975 7.975 0 0120 13a7.975 7.975 0 01-2.343 5.657z" />
                                    </svg>
                                </div>
                                <div>
                                    <p class="text-sm font-semibold text-main">Roasting Mode</p>
                                    <p class="text-xs text-muted">Savage responses</p>
                                </div>
                            </div>
                            <button type="button"
                                    onclick="toggleRoasting{{ $tenant->id }}()"
                                    class="relative inline-flex h-7 w-12 items-center rounded-full transition-colors duration-200 focus:outline-none focus:ring-2 focus:ring-accent-primary/50 {{ $roastingMode ? 'bg-orange-500' : 'bg-gray-600' }}"
                                    id="roasting-btn-{{ $tenant->id }}"
                                    role="switch"
                                    aria-checked="{{ $roastingMode ? 'true' : 'false' }}">
                                <span class="inline-block h-5 w-5 transform rounded-full bg-white shadow-lg transition-transform duration-200 {{ $roastingMode ? 'translate-x-6' : 'translate-x-1' }}"
                                      id="roasting-switch-{{ $tenant->id }}"></span>
                            </button>
                        </div>

                        <script>
                            let roastingMode{{ $tenant->id }} = {{ $roastingMode ? 'true' : 'false' }};

                            function toggleRoasting{{ $tenant->id }}() {
                                const url = '/tenants/{{ $tenant->id }}/settings/toggle/roasting_mode';

                                fetch(url, {
                                    method: 'POST',
                                    headers: {
                                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                        'Accept': 'application/json',
                                        'Content-Type': 'application/json'
                                    }
                                })
                                .then(response => response.json())
                                .then(data => {
                                    roastingMode{{ $tenant->id }} = !roastingMode{{ $tenant->id }};

                                    const btn = document.getElementById('roasting-btn-{{ $tenant->id }}');
                                    const switchEl = document.getElementById('roasting-switch-{{ $tenant->id }}');
                                    const icon = document.getElementById('roasting-icon-{{ $tenant->id }}');

                                    if (roastingMode{{ $tenant->id }}) {
                                        btn.classList.remove('bg-gray-600');
                                        btn.classList.add('bg-orange-500');
                                        btn.setAttribute('aria-checked', 'true');
                                        switchEl.classList.remove('translate-x-1');
                                        switchEl.classList.add('translate-x-6');
                                        icon.classList.remove('bg-surface', 'text-muted');
                                        icon.classList.add('bg-orange-500/15', 'text-orange-400');
                                    } else {
                                        btn.classList.add('bg-gray-600');
                                        btn.classList.remove('bg-orange-500');
                                        btn.setAttribute('aria-checked', 'false');
                                        switchEl.classList.add('translate-x-1');
                                        switchEl.classList.remove('translate-x-6');
                                        icon.classList.add('bg-surface', 'text-muted');
                                        icon.classList.remove('bg-orange-500/15', 'text-orange-400');
                                    }
                                })
                                .catch(error => {
                                    console.error('Error:', error);
                                    alert('Failed to toggle roasting mode');
                                });
                            }
                        </script>
                    </div>
                </div>
            </section>

            <!-- Error/Info Alert -->
            @if($tenant->last_error)
            @php
                $isWaitingForStream = str_contains($tenant->last_error, 'Waiting');
            @endphp

            @if($isWaitingForStream)
            <!-- Info Alert (Waiting for Stream) -->
            <section class="bg-blue-500/10 border border-blue-500/25 rounded-xl p-5">
                <div class="flex items-start gap-4">
                    <div class="flex-shrink-0 w-10 h-10 rounded-lg bg-blue-500/20 flex items-center justify-center">
                        <svg class="w-5 h-5 text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                    </div>
                    <div class="flex-1 min-w-0">
                        <h3 class="text-sm font-semibold text-blue-400 mb-1">Waiting for Stream</h3>
                        <p class="text-sm text-blue-300/80">{{ $tenant->last_error }}</p>
                        @if($tenant->last_error_at)
                        <p class="text-xs text-blue-400/60 mt-2">{{ \Carbon\Carbon::parse($tenant->last_error_at)->diffForHumans() }}</p>
                        @endif
                        <p class="text-xs text-blue-300/70 mt-3 flex items-center gap-1.5">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                            </svg>
                            Bot will automatically connect when you go live
                        </p>
                    </div>
                </div>
            </section>
            @else
            <!-- Error Alert -->
            <section class="bg-danger/10 border border-danger/25 rounded-xl p-5">
                <div class="flex flex-col lg:flex-row lg:items-start gap-4">
                    <div class="flex items-start gap-4 flex-1 min-w-0">
                        <div class="flex-shrink-0 w-10 h-10 rounded-lg bg-danger/20 flex items-center justify-center">
                            <svg class="w-5 h-5 text-danger" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
                            </svg>
                        </div>
                        <div class="flex-1 min-w-0">
                            <h3 class="text-sm font-semibold text-danger mb-1">Bot Error</h3>
                            <p class="text-sm text-red-300/80 break-words">{{ $tenant->last_error }}</p>
                            @if($tenant->last_error_at)
                            <p class="text-xs text-danger/60 mt-2">{{ \Carbon\Carbon::parse($tenant->last_error_at)->diffForHumans() }}</p>
                            @endif
                            @if(str_contains($tenant->last_error, 'OAuth') || str_contains($tenant->last_error, 'forbidden') || str_contains($tenant->last_error, 'scopes'))
                            <p class="text-xs text-red-300/80 mt-3">
                                <strong>Fix:</strong> Reconnect your YouTube channel with proper permissions.
                            </p>
                            @endif
                        </div>
                    </div>
                    @if(str_contains($tenant->last_error, 'OAuth') || str_contains($tenant->last_error, 'forbidden') || str_contains($tenant->last_error, 'scopes'))
                    <a href="{{ route('oauth.google.redirect') }}"
                       class="inline-flex items-center justify-center px-4 py-2.5 bg-white hover:bg-gray-50 text-gray-700 text-sm font-medium rounded-lg border border-gray-200 shadow-sm transition-all whitespace-nowrap">
                        <svg class="w-4 h-4 mr-2" viewBox="0 0 24 24">
                            <path fill="#4285F4" d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z"/>
                            <path fill="#34A853" d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z"/>
                            <path fill="#FBBC05" d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z"/>
                            <path fill="#EA4335" d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z"/>
                        </svg>
                        Reconnect
                    </a>
                    @endif
                </div>
            </section>
            @endif
            @endif

            <!-- =============================================== -->
            <!-- SECTION 2: Stats & Metrics Grid -->
            <!-- =============================================== -->
            <section>
                <h2 class="text-sm font-semibold text-muted uppercase tracking-wider mb-4 px-1">Live Metrics</h2>
                <div class="grid grid-cols-2 lg:grid-cols-4 gap-4">
                    <!-- Messages 1min -->
                    <div class="group bg-card border border-subtle rounded-xl p-5 hover:border-accent-primary/40 transition-all duration-200">
                        <div class="flex items-center justify-between mb-4">
                            <span class="text-xs font-semibold text-muted uppercase tracking-wider">Messages</span>
                            <div class="w-8 h-8 rounded-lg bg-accent-primary/10 flex items-center justify-center group-hover:bg-accent-primary/20 transition-colors">
                                <svg class="w-4 h-4 text-accent-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
                                </svg>
                            </div>
                        </div>
                        <div class="text-3xl lg:text-4xl font-bold text-main mb-1">{{ $stats['last_1min'] }}</div>
                        <p class="text-xs text-muted">Last minute</p>
                    </div>

                    <!-- Messages 5min -->
                    <div class="group bg-card border border-subtle rounded-xl p-5 hover:border-accent-secondary/40 transition-all duration-200">
                        <div class="flex items-center justify-between mb-4">
                            <span class="text-xs font-semibold text-muted uppercase tracking-wider">Messages</span>
                            <div class="w-8 h-8 rounded-lg bg-accent-secondary/10 flex items-center justify-center group-hover:bg-accent-secondary/20 transition-colors">
                                <svg class="w-4 h-4 text-accent-secondary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z" />
                                </svg>
                            </div>
                        </div>
                        <div class="text-3xl lg:text-4xl font-bold text-main mb-1">{{ $stats['last_5min'] }}</div>
                        <p class="text-xs text-muted">Last 5 minutes</p>
                    </div>

                    <!-- Unique Viewers -->
                    <div class="group bg-card border border-subtle rounded-xl p-5 hover:border-success/40 transition-all duration-200">
                        <div class="flex items-center justify-between mb-4">
                            <span class="text-xs font-semibold text-muted uppercase tracking-wider">Viewers</span>
                            <div class="w-8 h-8 rounded-lg bg-success/10 flex items-center justify-center group-hover:bg-success/20 transition-colors">
                                <svg class="w-4 h-4 text-success" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
                                </svg>
                            </div>
                        </div>
                        <div class="text-3xl lg:text-4xl font-bold text-main mb-1">{{ $stats['unique_viewers_1h'] }}</div>
                        <p class="text-xs text-muted">Unique (1 hour)</p>
                    </div>

                    <!-- Quota Usage -->
                    <div class="group bg-card border border-subtle rounded-xl p-5 hover:border-warning/40 transition-all duration-200">
                        <div class="flex items-center justify-between mb-4">
                            <span class="text-xs font-semibold text-muted uppercase tracking-wider">Quota</span>
                            <div class="w-8 h-8 rounded-lg bg-warning/10 flex items-center justify-center group-hover:bg-warning/20 transition-colors">
                                <svg class="w-4 h-4 text-warning" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                                </svg>
                            </div>
                        </div>
                        @php
                            $quotaPercent = $quota['per_minute']['limit'] > 0 ? round(($quota['per_minute']['used'] / $quota['per_minute']['limit']) * 100) : 0;
                        @endphp
                        <div class="text-3xl lg:text-4xl font-bold text-main mb-1">{{ $quotaPercent }}%</div>
                        <p class="text-xs text-muted">Per minute usage</p>
                        <!-- Progress Bar -->
                        <div class="mt-3 h-1.5 bg-surface rounded-full overflow-hidden">
                            <div class="h-full rounded-full transition-all duration-500 {{ $quotaPercent > 80 ? 'bg-danger' : ($quotaPercent > 50 ? 'bg-warning' : 'bg-success') }}"
                                 style="width: {{ min($quotaPercent, 100) }}%"></div>
                        </div>
                    </div>
                </div>
            </section>

            <!-- Stream Context (if available) -->
            @if($tenant->stream_type || $tenant->current_game)
            <section class="bg-gradient-to-r from-accent-secondary/10 to-accent-primary/10 border border-accent-secondary/20 rounded-xl p-5">
                <div class="flex items-center gap-4">
                    <div class="flex-shrink-0 w-12 h-12 rounded-xl bg-gradient-to-br from-accent-secondary/20 to-accent-primary/20 flex items-center justify-center">
                        <svg class="w-6 h-6 text-accent-secondary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z" />
                        </svg>
                    </div>
                    <div class="flex-1 min-w-0">
                        <div class="flex items-center gap-2 flex-wrap">
                            <span class="text-sm font-semibold text-main">Stream Context:</span>
                            @if($tenant->stream_type)
                            <span class="inline-flex items-center px-2.5 py-1 rounded-lg bg-accent-secondary/15 text-accent-secondary text-xs font-medium">
                                {{ ucfirst($tenant->stream_type) }}
                            </span>
                            @endif
                        </div>
                        @if($tenant->current_game)
                        <p class="text-sm text-secondary mt-1">
                            Playing <span class="text-accent-primary font-medium">{{ $tenant->current_game }}</span>
                        </p>
                        @endif
                    </div>
                </div>
            </section>
            @endif

            <!-- =============================================== -->
            <!-- SECTION 3: Feature Modules Grid -->
            <!-- =============================================== -->
            <section>
                <h2 class="text-sm font-semibold text-muted uppercase tracking-wider mb-4 px-1">Manage Bot</h2>
                <div class="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-5 gap-4">
                    <!-- Commands -->
                    <a href="{{ route('tenant.commands.index', $tenant) }}"
                       class="group relative bg-card border border-subtle rounded-xl p-5 hover:border-accent-primary/50 hover:bg-accent-primary/5 transition-all duration-200">
                        <div class="w-12 h-12 rounded-xl bg-accent-primary/10 flex items-center justify-center mb-4 group-hover:bg-accent-primary/20 group-hover:scale-110 transition-all duration-200">
                            <svg class="w-6 h-6 text-accent-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 9l3 3-3 3m5 0h3M5 20h14a2 2 0 002-2V6a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                            </svg>
                        </div>
                        <h3 class="text-sm font-semibold text-main mb-1">Commands</h3>
                        <p class="text-xs text-muted">Custom chat commands</p>
                        <svg class="absolute top-4 right-4 w-4 h-4 text-muted opacity-0 group-hover:opacity-100 transition-opacity" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7" />
                        </svg>
                    </a>

                    <!-- Timers -->
                    <a href="{{ route('tenant.timers.index', $tenant) }}"
                       class="group relative bg-card border border-subtle rounded-xl p-5 hover:border-accent-secondary/50 hover:bg-accent-secondary/5 transition-all duration-200">
                        <div class="w-12 h-12 rounded-xl bg-accent-secondary/10 flex items-center justify-center mb-4 group-hover:bg-accent-secondary/20 group-hover:scale-110 transition-all duration-200">
                            <svg class="w-6 h-6 text-accent-secondary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                            </svg>
                        </div>
                        <h3 class="text-sm font-semibold text-main mb-1">Timers</h3>
                        <p class="text-xs text-muted">Scheduled messages</p>
                        <svg class="absolute top-4 right-4 w-4 h-4 text-muted opacity-0 group-hover:opacity-100 transition-opacity" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7" />
                        </svg>
                    </a>

                    <!-- Banned Keywords -->
                    <a href="{{ route('tenant.banned-keywords.index', $tenant) }}"
                       class="group relative bg-card border border-subtle rounded-xl p-5 hover:border-warning/50 hover:bg-warning/5 transition-all duration-200">
                        <div class="w-12 h-12 rounded-xl bg-warning/10 flex items-center justify-center mb-4 group-hover:bg-warning/20 group-hover:scale-110 transition-all duration-200">
                            <svg class="w-6 h-6 text-warning" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M18.364 18.364A9 9 0 005.636 5.636m12.728 12.728A9 9 0 015.636 5.636m12.728 12.728L5.636 5.636" />
                            </svg>
                        </div>
                        <h3 class="text-sm font-semibold text-main mb-1">Ban Words</h3>
                        <p class="text-xs text-muted">Content moderation</p>
                        <svg class="absolute top-4 right-4 w-4 h-4 text-muted opacity-0 group-hover:opacity-100 transition-opacity" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7" />
                        </svg>
                    </a>

                    <!-- Stream Info -->
                    <button type="button"
                            x-data
                            @click="$dispatch('open-modal', 'stream-info-{{ $tenant->id }}')"
                            class="group relative bg-card border border-subtle rounded-xl p-5 hover:border-success/50 hover:bg-success/5 transition-all duration-200 text-left w-full">
                        <div class="w-12 h-12 rounded-xl bg-success/10 flex items-center justify-center mb-4 group-hover:bg-success/20 group-hover:scale-110 transition-all duration-200">
                            <svg class="w-6 h-6 text-success" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                            </svg>
                        </div>
                        <h3 class="text-sm font-semibold text-main mb-1">Stream Info</h3>
                        <p class="text-xs text-muted">Context & details</p>
                        <svg class="absolute top-4 right-4 w-4 h-4 text-muted opacity-0 group-hover:opacity-100 transition-opacity" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7" />
                        </svg>
                    </button>

                    <!-- Remove Channel -->
                    <form method="POST" action="{{ route('tenant.destroy', $tenant) }}" onsubmit="return confirm('Are you sure you want to remove this channel? This action cannot be undone.')">
                        @csrf
                        @method('DELETE')
                        <button type="submit"
                                class="group relative w-full bg-card border border-subtle rounded-xl p-5 hover:border-danger/50 hover:bg-danger/5 transition-all duration-200 text-left">
                            <div class="w-12 h-12 rounded-xl bg-danger/10 flex items-center justify-center mb-4 group-hover:bg-danger/20 group-hover:scale-110 transition-all duration-200">
                                <svg class="w-6 h-6 text-danger" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                </svg>
                            </div>
                            <h3 class="text-sm font-semibold text-danger mb-1">Remove</h3>
                            <p class="text-xs text-danger/70">Delete channel</p>
                        </button>
                    </form>
                </div>
            </section>
        </div>

        <!-- Stream Info Modal -->
        <div x-data="{ open: false }"
             @open-modal.window="open = ($event.detail === 'stream-info-{{ $tenant->id }}')"
             @close-modal.window="open = false"
             @keydown.escape.window="open = false"
             x-show="open"
             x-transition:enter="ease-out duration-200"
             x-transition:enter-start="opacity-0"
             x-transition:enter-end="opacity-100"
             x-transition:leave="ease-in duration-150"
             x-transition:leave-start="opacity-100"
             x-transition:leave-end="opacity-0"
             class="fixed inset-0 z-50 overflow-y-auto"
             style="display: none;">
            <div class="flex items-center justify-center min-h-screen px-4 py-8">
                <!-- Backdrop -->
                <div class="fixed inset-0 bg-black/70 backdrop-blur-sm" @click="open = false"></div>

                <!-- Modal Content -->
                <div x-transition:enter="ease-out duration-200"
                     x-transition:enter-start="opacity-0 scale-95"
                     x-transition:enter-end="opacity-100 scale-100"
                     x-transition:leave="ease-in duration-150"
                     x-transition:leave-start="opacity-100 scale-100"
                     x-transition:leave-end="opacity-0 scale-95"
                     class="relative bg-card border border-subtle rounded-2xl shadow-2xl max-w-xl w-full">
                    <!-- Modal Header -->
                    <div class="flex items-center justify-between px-6 py-4 border-b border-subtle">
                        <div>
                            <h3 class="text-lg font-semibold text-main">Stream Info</h3>
                            <p class="text-sm text-muted">Configure bot context for better responses</p>
                        </div>
                        <button @click="open = false" class="p-2 text-muted hover:text-main rounded-lg hover:bg-surface transition-colors">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                            </svg>
                        </button>
                    </div>

                    <!-- Modal Body -->
                    <form method="POST" action="{{ route('tenant.stream-info.update', $tenant) }}" class="p-6 space-y-5">
                        @csrf

                        <div>
                            <label class="block text-sm font-medium text-secondary mb-2">Stream Type</label>
                            <select name="stream_type" class="w-full px-4 py-3 bg-surface border border-subtle rounded-xl text-main focus:outline-none focus:ring-2 focus:ring-accent-primary/50 focus:border-accent-primary transition-colors">
                                <option value="">Select stream type...</option>
                                <option value="gaming" {{ $tenant->stream_type === 'gaming' ? 'selected' : '' }}>Gaming</option>
                                <option value="tech" {{ $tenant->stream_type === 'tech' ? 'selected' : '' }}>Tech / Coding</option>
                                <option value="music" {{ $tenant->stream_type === 'music' ? 'selected' : '' }}>Music</option>
                                <option value="cooking" {{ $tenant->stream_type === 'cooking' ? 'selected' : '' }}>Cooking</option>
                                <option value="fitness" {{ $tenant->stream_type === 'fitness' ? 'selected' : '' }}>Fitness</option>
                                <option value="vlog" {{ $tenant->stream_type === 'vlog' ? 'selected' : '' }}>Vlog</option>
                                <option value="education" {{ $tenant->stream_type === 'education' ? 'selected' : '' }}>Education</option>
                                <option value="other" {{ $tenant->stream_type === 'other' ? 'selected' : '' }}>Other</option>
                            </select>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-secondary mb-2">Current Game <span class="text-muted font-normal">(optional)</span></label>
                            <input type="text" name="current_game" value="{{ $tenant->current_game }}"
                                   placeholder="e.g., GTA V, Valorant, Minecraft"
                                   class="w-full px-4 py-3 bg-surface border border-subtle rounded-xl text-main placeholder:text-muted focus:outline-none focus:ring-2 focus:ring-accent-primary/50 focus:border-accent-primary transition-colors">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-secondary mb-2">Personal Info <span class="text-muted font-normal">(any language)</span></label>
                            <textarea name="personal_info" rows="5"
                                      placeholder="Tell viewers about yourself..."
                                      class="w-full px-4 py-3 bg-surface border border-subtle rounded-xl text-main placeholder:text-muted focus:outline-none focus:ring-2 focus:ring-accent-primary/50 focus:border-accent-primary transition-colors resize-none">{{ $tenant->personal_info }}</textarea>
                            <p class="text-xs text-muted mt-2">Bot uses this to answer questions about you (name, location, etc.)</p>
                        </div>

                        <!-- Modal Footer -->
                        <div class="flex items-center justify-end gap-3 pt-4 border-t border-subtle">
                            <button type="button" @click="open = false"
                                    class="px-5 py-2.5 bg-surface hover:bg-surface-hover text-secondary hover:text-main border border-subtle rounded-xl text-sm font-medium transition-colors">
                                Cancel
                            </button>
                            <button type="submit"
                                    class="px-6 py-2.5 bg-accent-primary hover:bg-purple-600 text-white text-sm font-semibold rounded-xl shadow-lg shadow-accent-primary/25 hover:shadow-xl transition-all">
                                Save Changes
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        @endforeach

    </div>
</div>
@endsection
