<?php

use App\Http\Controllers\Admin\AdminController;
use App\Http\Controllers\Admin\SettingsController;
use App\Http\Controllers\Auth\BotAccountOAuthController;
use App\Http\Controllers\Auth\GoogleOAuthController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\PageController;
use App\Http\Controllers\Tenant\CommandController;
use App\Http\Controllers\Tenant\TenantController;
use App\Http\Controllers\Tenant\TimerController;
use Illuminate\Support\Facades\Route;

// Public routes
Route::get('/', function () {
    return view('welcome');
})->name('welcome');

Route::get('/health', function () {
    return response()->json([
        'status' => 'ok',
        'timestamp' => now()->toIso8601String(),
    ]);
})->name('health');

// Policy pages (public)
Route::get('/privacy-policy', [PageController::class, 'privacyPolicy'])->name('privacy-policy');
Route::get('/terms-of-service', [PageController::class, 'termsOfService'])->name('terms-of-service');
Route::get('/contact', [PageController::class, 'contact'])->name('contact');

// Auth routes (Breeze)
require __DIR__.'/auth.php';

// Test route
require __DIR__.'/test-admin.php';

// Authenticated routes - All users can access
Route::middleware(['auth', 'verified'])->group(function () {
    // Dashboard - All authenticated users
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');

    // Bot Account OAuth (Admin Only - vanihbx@gmail.com)
    Route::middleware(\App\Http\Middleware\EnsureUserIsAuthorized::class)->prefix('bot-account')->name('bot-account.')->group(function () {
        Route::get('oauth/redirect', [BotAccountOAuthController::class, 'redirect'])->name('oauth.redirect');
        Route::get('oauth/callback', [BotAccountOAuthController::class, 'callback'])->name('oauth.callback');
        Route::post('disconnect', [BotAccountOAuthController::class, 'disconnect'])->name('disconnect');
    });

    // Tenant management - All users can manage their own tenants
    Route::prefix('tenants/{tenant}')->name('tenant.')->group(function () {
        Route::post('start', [TenantController::class, 'start'])->name('start');
        Route::post('stop', [TenantController::class, 'stop'])->name('stop');
        Route::delete('/', [TenantController::class, 'destroy'])->name('destroy');
        Route::post('settings/toggle/{key}', [TenantController::class, 'toggleSetting'])->name('settings.toggle');
        Route::post('settings', [TenantController::class, 'updateSettings'])->name('settings.update');
        Route::post('stream-info', [TenantController::class, 'updateStreamInfo'])->name('stream-info.update');
        Route::post('spam-protection', [TenantController::class, 'updateSpamProtection'])->name('spam-protection.update');

        // Commands
        Route::resource('commands', \App\Http\Controllers\Tenant\CommandController::class)->except(['show']);

        // Timers
        Route::resource('timers', \App\Http\Controllers\Tenant\TimerController::class)->except(['show']);

        // Banned Keywords
        Route::get('banned-keywords', [\App\Http\Controllers\Tenant\BannedKeywordController::class, 'index'])->name('banned-keywords.index');
        Route::post('banned-keywords', [\App\Http\Controllers\Tenant\BannedKeywordController::class, 'store'])->name('banned-keywords.store');
        Route::put('banned-keywords/{bannedKeyword}', [\App\Http\Controllers\Tenant\BannedKeywordController::class, 'update'])->name('banned-keywords.update');
        Route::delete('banned-keywords/{bannedKeyword}', [\App\Http\Controllers\Tenant\BannedKeywordController::class, 'destroy'])->name('banned-keywords.destroy');
        Route::post('banned-keywords/{bannedKeyword}/toggle', [\App\Http\Controllers\Tenant\BannedKeywordController::class, 'toggle'])->name('banned-keywords.toggle');
    });

    // User Settings
    Route::post('change-password', [\App\Http\Controllers\User\ChangePasswordController::class, 'update'])->name('password.change');

    // Admin routes - ONLY for vanihbx@gmail.com
    Route::middleware(\App\Http\Middleware\EnsureUserIsAuthorized::class)->prefix('admin')->name('admin.')->group(function () {
        Route::get('/', [AdminController::class, 'index'])->name('index');
        Route::post('tenants/{tenant}/force-stop', [AdminController::class, 'forceStop'])->name('tenants.force-stop');
        Route::post('tenants/{tenant}/force-start', [AdminController::class, 'forceStart'])->name('tenants.force-start');
        Route::get('tenants/{tenant}/logs', [AdminController::class, 'logs'])->name('tenants.logs');
        Route::get('audits', [AdminController::class, 'audits'])->name('audits');
        Route::get('quotas', [AdminController::class, 'quotas'])->name('quotas');
        Route::post('tenants/{tenant}/quota', [AdminController::class, 'updateQuota'])->name('tenants.quota');

        // API Settings
        Route::get('settings', [SettingsController::class, 'index'])->name('settings');
        Route::post('settings/openai', [SettingsController::class, 'updateOpenAI'])->name('settings.openai');
        Route::post('settings/google', [SettingsController::class, 'updateGoogle'])->name('settings.google');
        Route::post('settings/test-openai', [SettingsController::class, 'testOpenAI'])->name('settings.test-openai');
    });
});
