<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Http\Requests\Timer\StoreTimerRequest;
use App\Http\Requests\Timer\UpdateTimerRequest;
use App\Models\Audit;
use App\Models\Tenant;
use App\Models\Timer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class TimerController extends Controller
{
    public function index(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $timers = $tenant->timers()->paginate(20);

        return view('tenant.timers.index', compact('tenant', 'timers'));
    }

    public function create(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        return view('tenant.timers.create', compact('tenant'));
    }

    public function store(StoreTimerRequest $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $timer = $tenant->timers()->create($request->validated());

        Audit::log($tenant->id, $request->user()->id, 'timer_created', [
            'timer_id' => $timer->id,
        ]);

        return redirect()->route('tenant.timers.index', $tenant)
            ->with('success', 'Timer created successfully!');
    }

    public function edit(Request $request, Tenant $tenant, Timer $timer)
    {
        Gate::authorize('manage', $tenant);

        if ($timer->tenant_id !== $tenant->id) {
            abort(404);
        }

        return view('tenant.timers.edit', compact('tenant', 'timer'));
    }

    public function update(UpdateTimerRequest $request, Tenant $tenant, Timer $timer)
    {
        Gate::authorize('manage', $tenant);

        if ($timer->tenant_id !== $tenant->id) {
            abort(404);
        }

        $timer->update($request->validated());

        Audit::log($tenant->id, $request->user()->id, 'timer_updated', [
            'timer_id' => $timer->id,
        ]);

        return redirect()->route('tenant.timers.index', $tenant)
            ->with('success', 'Timer updated successfully!');
    }

    public function destroy(Request $request, Tenant $tenant, Timer $timer)
    {
        Gate::authorize('manage', $tenant);

        if ($timer->tenant_id !== $tenant->id) {
            abort(404);
        }

        $timer->delete();

        Audit::log($tenant->id, $request->user()->id, 'timer_deleted');

        return redirect()->route('tenant.timers.index', $tenant)
            ->with('success', 'Timer deleted successfully.');
    }
}
