<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Tenant extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'channel_id',
        'channel_title',
        'channel_handle',
        'live_chat_id',
        'active_broadcast_id',
        'status',
        'last_seen_message_id',
        'last_polled_at',
        'broadcast_started_at',
        'quotas',
        'settings',
        'last_error',
        'last_error_at',
    ];

    protected function casts(): array
    {
        return [
            'last_polled_at' => 'datetime',
            'broadcast_started_at' => 'datetime',
            'last_error_at' => 'datetime',
            'quotas' => 'array',
            'settings' => 'array',
        ];
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function oauthTokens()
    {
        return $this->hasMany(OauthToken::class);
    }

    public function commands()
    {
        return $this->hasMany(Command::class);
    }

    public function timers()
    {
        return $this->hasMany(Timer::class);
    }

    public function filters()
    {
        return $this->hasMany(Filter::class);
    }

    public function giveaways()
    {
        return $this->hasMany(Giveaway::class);
    }

    public function messagesLog()
    {
        return $this->hasMany(MessageLog::class);
    }

    public function roleOverrides()
    {
        return $this->hasMany(RoleOverride::class);
    }

    public function tenantSettings()
    {
        return $this->hasMany(Setting::class);
    }

    public function audits()
    {
        return $this->hasMany(Audit::class);
    }

    public function isRunning(): bool
    {
        return $this->status === 'running';
    }

    public function isStopped(): bool
    {
        return $this->status === 'stopped';
    }

    public function hasError(): bool
    {
        return $this->status === 'error';
    }

    public function getUptime(): ?int
    {
        if (!$this->broadcast_started_at) {
            return null;
        }

        return now()->diffInSeconds($this->broadcast_started_at);
    }

    public function getSetting(string $key, $default = null)
    {
        $setting = $this->tenantSettings()->where('key', $key)->first();
        return $setting ? $setting->value : $default;
    }
}
