<?php

use App\Models\Command;
use App\Models\Tenant;
use App\Services\ActivityHeuristics;
use App\Services\CommandRuntime;

beforeEach(function () {
    $this->activityHeuristics = Mockery::mock(ActivityHeuristics::class);
    $this->commandRuntime = new CommandRuntime($this->activityHeuristics);
});

test('executes built-in help command', function () {
    $tenant = createTenant();
    $response = $this->commandRuntime->execute($tenant, '!help', []);

    expect($response)->toContain('PXEBot commands');
});

test('executes built-in ping command', function () {
    $tenant = createTenant();
    $response = $this->commandRuntime->execute($tenant, '!ping', []);

    expect($response)->toBe('Pong! 🏓 PXEBot is online and ready.');
});

test('executes custom command', function () {
    $tenant = createTenant();
    $command = createCommand($tenant, [
        'name' => 'discord',
        'response' => 'Join us at discord.gg/example',
        'is_enabled' => true,
    ]);

    $response = $this->commandRuntime->execute($tenant, '!discord', [
        'tenant_id' => $tenant->id,
        'author_id' => 'test_author',
        'author_name' => 'TestUser',
        'channel_name' => $tenant->channel_title,
    ]);

    expect($response)->toBe('Join us at discord.gg/example');
});

test('returns null for non-command text', function () {
    $tenant = createTenant();
    $response = $this->commandRuntime->execute($tenant, 'just a regular message', []);

    expect($response)->toBeNull();
});

test('expands user variable in command', function () {
    $tenant = createTenant();
    $command = createCommand($tenant, [
        'name' => 'welcome',
        'response' => 'Welcome {user} to the stream!',
        'is_enabled' => true,
    ]);

    $response = $this->commandRuntime->execute($tenant, '!welcome', [
        'tenant_id' => $tenant->id,
        'author_id' => 'test_author',
        'author_name' => 'JohnDoe',
        'channel_name' => $tenant->channel_title,
    ]);

    expect($response)->toBe('Welcome JohnDoe to the stream!');
});
