# PXEBot

**Production-ready Nightbot-style YouTube Live Chat Bot with OpenAI Integration**

PXEBot is a complete, multi-tenant YouTube chatbot solution built with Laravel 11. It provides Nightbot-like features (custom commands, timers, moderation filters, giveaways) plus OpenAI-powered multilingual Q&A that automatically replies in the same language as the user's message.

---

## Features

### Core Capabilities
- ✅ **Multi-Tenant Architecture** - Each YouTube channel is an isolated tenant with its own config, commands, and quotas
- ✅ **YouTube Live Chat Integration** - Real-time polling, adaptive backoff, token refresh, quota management
- ✅ **OpenAI Multilingual Answers** - Automatically detects language and replies in the same language
- ✅ **Custom Commands** - Create unlimited commands with cooldowns, role permissions, aliases, and variable expansion
- ✅ **Built-in Commands** - `!help`, `!commands`, `!uptime`, `!discord`, `!lurk`, `!so`, `!rules`, `!ping`
- ✅ **Timers** - Scheduled messages with chat activity checks
- ✅ **Moderation Filters** - Caps, symbols, links, blacklist words, emoji flood detection
- ✅ **Giveaways** - Start/cancel, keyword entry, random winner selection
- ✅ **Role-Based Permissions** - Owner, Admin, Mod, Trusted, Viewer
- ✅ **Rate Limiting** - Per-tenant sliding window rate limits (Redis-backed)
- ✅ **User & Admin Panels** - Dashboard for bot control, CRUD operations, logs, and stats
- ✅ **Audit Logging** - Track all significant actions

### Tech Stack
- **Backend**: PHP 8.2+, Laravel 11
- **Database**: MySQL 8 (InnoDB, utf8mb4)
- **Cache/Queue**: Redis + Laravel Horizon
- **Auth**: Laravel Breeze + Google OAuth2
- **UI**: Blade Templates + TailwindCSS + Alpine.js
- **Testing**: Pest (feature + unit tests)
- **Deployment**: Docker + docker-compose + Nginx

---

## Prerequisites

- **Docker** and **Docker Compose** (recommended for local dev)
- OR: **PHP 8.2+**, **Composer**, **MySQL 8**, **Redis**, **Node.js/NPM**
- **Google Cloud Project** with YouTube Data API v3 enabled
- **OpenAI API Key**

---

## Quick Start (Local Development with Docker)

### 1. Clone and Setup

```bash
cd pxebots
cp .env.example .env
```

### 2. Configure Environment

Edit `.env` with your credentials:

```bash
# Database
DB_DATABASE=pxebot
DB_USERNAME=laravel
DB_PASSWORD=secret

# YouTube OAuth2 (from Google Cloud Console)
YT_CLIENT_ID=your-client-id.apps.googleusercontent.com
YT_CLIENT_SECRET=your-client-secret
YT_REDIRECT_URI=http://localhost/oauth/google/callback

# OpenAI
OPENAI_API_KEY=sk-proj-your-key-here
OPENAI_MODEL=gpt-4o-mini
```

### 3. Start Docker Services

```bash
docker-compose up -d
```

This starts:
- `app` - PHP-FPM + Nginx (port 80)
- `mysql` - MySQL 8 (port 3306)
- `redis` - Redis (port 6379)
- `horizon` - Laravel Horizon queue worker
- `scheduler` - Laravel scheduler

### 4. Install Dependencies & Migrate

```bash
docker-compose exec app composer install
docker-compose exec app php artisan key:generate
docker-compose exec app php artisan migrate --seed
docker-compose exec app php artisan storage:link
```

### 5. Access the Application

- **App**: http://localhost
- **Horizon Dashboard**: http://localhost/horizon

### 6. Register & Connect YouTube

1. Register a new account at http://localhost/register
2. Log in and go to Dashboard
3. Click **"Connect YouTube Channel"**
4. Authorize with Google OAuth2
5. Click **"Start Bot"** to begin polling

---

## Google Cloud Console Setup

### 1. Create Project & Enable API

1. Go to [Google Cloud Console](https://console.cloud.google.com/)
2. Create a new project (e.g., "PXEBot")
3. Enable **YouTube Data API v3**

### 2. Create OAuth2 Credentials

1. Navigate to **APIs & Services > Credentials**
2. Click **+ CREATE CREDENTIALS > OAuth client ID**
3. Application type: **Web application**
4. Name: `PXEBot`
5. Authorized redirect URIs:
   ```
   http://localhost/oauth/google/callback
   https://your-production-domain.com/oauth/google/callback
   ```
6. Save and copy **Client ID** and **Client Secret** to `.env`

### 3. Configure OAuth Consent Screen

1. Go to **OAuth consent screen**
2. User Type: **External** (for public use) or **Internal** (for workspace)
3. Add required scopes:
   - `https://www.googleapis.com/auth/youtube.readonly`
   - `https://www.googleapis.com/auth/youtube.force-ssl`
   - `https://www.googleapis.com/auth/youtube`
4. Add test users (if in testing mode)

---

## OpenAI Setup

1. Get API key from [OpenAI Platform](https://platform.openai.com/api-keys)
2. Add to `.env`:
   ```
   OPENAI_API_KEY=sk-proj-your-key
   OPENAI_MODEL=gpt-4o-mini
   ```
3. Models:
   - **gpt-4o-mini** (recommended) - Cost-efficient, fast
   - **gpt-4o** - Higher quality, more expensive

### OpenAI Behavior

- **Language Detection**: Automatically detects message language
- **Same-Language Replies**: System prompt enforces replying in detected language
- **Content Moderation**: Optional pre-check before posting (enabled by default)
- **Question Detection**: Heuristic + LLM classification for auto-response triggers

---

## Architecture Overview

### Request Flow

1. **Scheduler** triggers `PollYouTubeChat` job every 2 seconds for running tenants
2. **PollYouTubeChat** fetches new messages from YouTube Live Chat API
3. Each message dispatches **ProcessIncomingMessage** job
4. **ProcessIncomingMessage**:
   - Checks filters (caps, links, blacklist, etc.)
   - Checks for giveaway entry
   - Tries to execute command (built-in or custom)
   - If no command, classifies message with OpenAI
   - If question detected, generates OpenAI reply
   - Dispatches **PostBotReply** job
5. **PostBotReply** sends message to YouTube respecting rate limits

### Database Schema

- `users` - App users (channel owners)
- `tenants` - YouTube channels (1 per user)
- `oauth_tokens` - Google OAuth tokens with refresh
- `commands` - Custom commands
- `timers` - Scheduled messages
- `filters` - Moderation rules
- `giveaways` + `giveaway_entries` - Giveaway system
- `messages_log` - All chat messages processed
- `role_overrides` - Manual role assignments
- `settings` - Tenant-specific settings
- `audits` - Action logging

---

## Usage

### Custom Commands

#### Create Command via UI
1. Dashboard → Commands → Add Command
2. Fill in:
   - **Name**: `discord` (will become `!discord`)
   - **Response**: `Join our Discord: https://discord.gg/example`
   - **Cooldown**: `10` seconds
   - **Required Role**: `viewer`
   - **Enabled**: ✅

#### Variable Expansion

Use these variables in command responses:
- `{user}` - Message author's name
- `{channel}` - Channel name
- `{query}` - Text after command
- `{uptime}` - Stream uptime
- `{count}` - Command use count
- `{random}` - Random number 1-100
- `{subscribers}` - Subscriber count
- `{viewers}` - Current viewers

Example:
```
Command: !welcome
Response: Welcome {user} to {channel}! We've been live for {uptime}!
```

### Timers

Timers fire only if chat has been active (configurable threshold).

#### Create Timer
1. Dashboard → Timers → Add Timer
2. Configuration:
   - **Name**: Reminder to subscribe
   - **Message**: Don't forget to subscribe! 🔔
   - **Interval**: `10` minutes
   - **Activity Window**: `5` minutes
   - **Enabled**: ✅

Timer will fire every ~10 minutes ±10% if ≥3 messages in last 5 minutes.

### Filters

Create moderation filters to automatically warn/timeout/ignore messages.

#### Filter Types
- **Caps**: Excessive uppercase (% threshold)
- **Symbols**: Too many non-alphanumeric chars (% threshold)
- **Links**: Any URL detected
- **Blacklist**: List of banned words/phrases
- **Emoji**: Emoji flood (count threshold)

#### Actions
- **Warn**: Post warning message
- **Timeout**: Filter silently (log but no action)
- **Ignore**: Block message processing

### Giveaways

1. Dashboard → Giveaways → Start New
2. Set keyword (e.g., `!enter`)
3. Set winner count
4. Users type `!enter` in chat to join
5. Click "Pick Winners" to randomly select

---

## Artisan Commands

### Bot Control

```bash
# Start all bots
php artisan bot:control start

# Stop all bots
php artisan bot:control stop

# Start specific tenant
php artisan bot:control start --tenant=1

# Check status
php artisan bot:control status
```

### Other Commands

```bash
# Run migrations
php artisan migrate

# Seed demo data
php artisan db:seed

# Run tests
php artisan test
# OR with Pest
./vendor/bin/pest

# Queue worker (dev)
php artisan queue:work redis --tries=3

# Horizon (production)
php artisan horizon

# Scheduler (requires cron in production, or use schedule:work for dev)
php artisan schedule:work
```

---

## Testing

### Run All Tests

```bash
php artisan test
```

OR with Pest directly:

```bash
./vendor/bin/pest
```

### Test Structure

- **Unit Tests**: `tests/Unit/` - Service logic (CommandRuntime, RateLimiter)
- **Feature Tests**: `tests/Feature/` - HTTP routes, authentication, authorization

### Example Tests

```php
// tests/Unit/CommandRuntimeTest.php
test('executes built-in help command', function () {
    $tenant = createTenant();
    $response = $this->commandRuntime->execute($tenant, '!help', []);
    expect($response)->toContain('PXEBot commands');
});

// tests/Feature/DashboardTest.php
test('authenticated user can access dashboard', function () {
    $user = createUser();
    $response = $this->actingAs($user)->get('/dashboard');
    $response->assertOk();
});
```

---

## Production Deployment

### 1. Server Requirements

- Ubuntu 20.04+ or similar
- Docker + Docker Compose
- Domain with SSL (e.g., Let's Encrypt)

### 2. Build & Deploy

```bash
# Clone repo
git clone https://github.com/yourorg/pxebot.git
cd pxebot

# Copy and configure .env
cp .env.example .env
nano .env  # Set production values

# Build production image
docker-compose -f docker-compose.prod.yml build

# Start services
docker-compose -f docker-compose.prod.yml up -d

# Run migrations
docker-compose exec app php artisan migrate --force
docker-compose exec app php artisan config:cache
docker-compose exec app php artisan route:cache
docker-compose exec app php artisan view:cache
```

### 3. Cron for Scheduler

Add to crontab:

```cron
* * * * * cd /path/to/pxebot && docker-compose exec -T app php artisan schedule:run >> /dev/null 2>&1
```

### 4. Supervisor for Horizon

Supervisor config is included in `docker/php/supervisord.conf`. Horizon starts automatically in Docker.

### 5. Nginx Reverse Proxy (if not using Docker Nginx)

```nginx
server {
    listen 80;
    server_name your-domain.com;

    location / {
        proxy_pass http://localhost:8000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
    }
}
```

### 6. SSL with Let's Encrypt

```bash
sudo apt install certbot python3-certbot-nginx
sudo certbot --nginx -d your-domain.com
```

---

## Configuration Reference

### Rate Limits

- `RATE_LIMIT_OUTBOUND_PER_MIN` (default: 12) - Max messages per minute per tenant
- `RATE_LIMIT_OUTBOUND_PER_HOUR` (default: 200) - Max messages per hour per tenant
- `RATE_LIMIT_USER_COOLDOWN` (default: 10) - Seconds between user messages

### YouTube Polling

- `YT_POLL_INTERVAL_SECONDS` (default: 2) - Base polling interval
- `YT_POLL_BACKOFF_MULTIPLIER` (default: 1.5) - Backoff on error
- `YT_POLL_MAX_INTERVAL_SECONDS` (default: 30) - Max backoff interval
- `YT_POLL_MAX_RESULTS` (default: 200) - Messages per poll

### OpenAI

- `OPENAI_MODEL` - Model to use (`gpt-4o-mini`, `gpt-4o`)
- `OPENAI_TEMPERATURE` (default: 0.7) - Response creativity (0.0-2.0)
- `OPENAI_MAX_TOKENS` (default: 200) - Max tokens per response
- `ENABLE_CONTENT_MODERATION` (default: true) - Pre-check messages
- `AUTO_DETECT_QUESTIONS` (default: true) - Auto-respond to questions

---

## API Quotas & Best Practices

### YouTube API Quota

- **Default**: 10,000 units/day
- **LiveChatMessages.list**: ~5 units per call
- **LiveChatMessages.insert**: ~50 units per call

**Best Practices**:
- Poll every 2-3 seconds (respect `pollingIntervalMillis` from API)
- Use rate limiting to avoid quota exhaustion
- Monitor quota usage in Google Cloud Console

### OpenAI Rate Limits

- **gpt-4o-mini**: 10,000 RPM, 200M TPM (Tier 1)
- **gpt-4o**: 500 RPM, 30K TPM (Tier 1)

**Best Practices**:
- Use `gpt-4o-mini` for production
- Implement user cooldowns
- Cache common responses if possible

---

## Troubleshooting

### Bot Not Polling

1. Check tenant status: `php artisan bot:control status`
2. Check Horizon dashboard: http://localhost/horizon
3. Verify OAuth token is valid (check `oauth_tokens` table)
4. Check `tenants.last_error` for error messages

### YouTube API 403 Forbidden

- Verify OAuth scopes in Google Cloud Console
- Ensure YouTube Data API v3 is enabled
- Check `oauth_tokens.scopes` in database

### OpenAI Errors

- Verify API key in `.env`
- Check OpenAI account quota/billing
- Review logs: `storage/logs/laravel.log`

### Rate Limit Issues

- Check Redis: `redis-cli get "rate_limit:outbound:min:1"`
- Reset limits: `php artisan tinker` → `app(RateLimiterService::class)->reset($tenant)`

### Scheduler Not Running

- Ensure cron is configured in production
- OR use `php artisan schedule:work` for dev
- Check `horizon:snapshot` is running every 5 minutes

---

## Project Structure

```
pxebot/
├── app/
│   ├── Console/
│   │   ├── Commands/
│   │   │   └── BotControl.php
│   │   └── Kernel.php          # Scheduler config
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── Admin/
│   │   │   ├── Auth/
│   │   │   └── Tenant/
│   │   ├── Requests/           # Form validation
│   │   └── Middleware/
│   ├── Jobs/
│   │   ├── PollYouTubeChat.php
│   │   ├── ProcessIncomingMessage.php
│   │   ├── PostBotReply.php
│   │   ├── FireTimer.php
│   │   └── RefreshYouTubeToken.php
│   ├── Models/
│   ├── Policies/
│   ├── Services/
│   │   ├── YouTubeService.php
│   │   ├── OpenAIService.php
│   │   ├── CommandRuntime.php
│   │   ├── RateLimiterService.php
│   │   └── ActivityHeuristics.php
│   └── Providers/
├── config/
│   ├── services.php            # Google, OpenAI, Bot config
│   └── horizon.php             # Queue config
├── database/
│   ├── migrations/
│   ├── factories/
│   └── seeders/
├── resources/
│   └── views/
│       ├── dashboard/
│       ├── tenant/
│       └── admin/
├── routes/
│   ├── web.php                 # Main routes
│   └── auth.php                # Auth routes
├── tests/
│   ├── Feature/
│   └── Unit/
├── docker/
│   ├── nginx/
│   └── php/
├── .env.example
├── composer.json
├── docker-compose.yml
├── Makefile
└── README.md
```

---

## Contributing

Contributions welcome! Please:

1. Fork the repo
2. Create a feature branch
3. Write tests for new features
4. Submit a pull request

---

## License

MIT License - see LICENSE file for details.

---

## Support

- **Issues**: https://github.com/yourorg/pxebot/issues
- **Discussions**: https://github.com/yourorg/pxebot/discussions
- **Email**: support@phxbot.com

---

## Credits

Built with:
- [Laravel](https://laravel.com)
- [OpenAI PHP](https://github.com/openai-php/laravel)
- [Laravel Horizon](https://laravel.com/docs/horizon)
- [Laravel Breeze](https://laravel.com/docs/breeze)
- [TailwindCSS](https://tailwindcss.com)
- [Alpine.js](https://alpinejs.dev)
- [Pest PHP](https://pestphp.com)

---

**Made with ❤️ for the YouTube creator community**

Domain: **phxbot.com**
