# 🚀 PXEBot Setup Guide

You have **two options** to run PXEBot:
1. **With Docker** (Recommended - Easier)
2. **Without Docker** (Local PHP/MySQL/Redis)

---

## Option 1: Setup with Docker (Recommended)

### Prerequisites
- Docker Desktop installed
- Docker Desktop running

### Step 1: Start Docker Desktop

**On macOS:**
1. Open **Docker Desktop** application
2. Wait for it to fully start (whale icon in menu bar should be steady)
3. Verify it's running:
   ```bash
   docker --version
   docker ps
   ```

**On Windows:**
1. Open **Docker Desktop** from Start Menu
2. Wait for "Docker Desktop is running" status
3. Verify in PowerShell:
   ```powershell
   docker --version
   docker ps
   ```

### Step 2: Setup Environment

```bash
cd /Users/cyb3rdog/Downloads/pxebots

# Copy environment file
cp .env.example .env

# Edit .env with your credentials (see below)
nano .env
```

**Required .env values:**
```env
# App
APP_KEY=  # Will be generated

# Database
DB_DATABASE=pxebot
DB_USERNAME=laravel
DB_PASSWORD=secret

# Google OAuth (from Google Cloud Console)
YT_CLIENT_ID=your-client-id.apps.googleusercontent.com
YT_CLIENT_SECRET=your-secret
YT_REDIRECT_URI=http://localhost/oauth/google/callback

# OpenAI
OPENAI_API_KEY=sk-proj-your-key
OPENAI_MODEL=gpt-4o-mini
```

### Step 3: Build and Start

```bash
# Build Docker images (first time only)
docker-compose build

# Start all services
docker-compose up -d

# Install PHP dependencies
docker-compose exec app composer install

# Generate app key
docker-compose exec app php artisan key:generate

# Run migrations
docker-compose exec app php artisan migrate

# Link storage
docker-compose exec app php artisan storage:link
```

### Step 4: Access Application

- **App**: http://localhost
- **Horizon**: http://localhost/horizon

### Common Docker Commands

```bash
# View logs
docker-compose logs -f app

# Stop services
docker-compose down

# Restart services
docker-compose restart

# Run artisan commands
docker-compose exec app php artisan [command]

# Access MySQL
docker-compose exec mysql mysql -u laravel -psecret pxebot
```

---

## Option 2: Setup without Docker (Local)

### Prerequisites

**On macOS (using Homebrew):**
```bash
# Install Homebrew if not installed
/bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"

# Install PHP 8.2
brew install php@8.2

# Install Composer
brew install composer

# Install MySQL
brew install mysql

# Install Redis
brew install redis

# Start services
brew services start mysql
brew services start redis
```

**On Windows:**
- Install XAMPP or Laragon (includes PHP, MySQL)
- Install Composer from https://getcomposer.org/download/
- Install Redis from https://github.com/microsoftarchive/redis/releases

**On Linux (Ubuntu/Debian):**
```bash
# Install PHP 8.2
sudo apt update
sudo apt install php8.2 php8.2-cli php8.2-fpm php8.2-mysql php8.2-redis \
    php8.2-mbstring php8.2-xml php8.2-bcmath php8.2-curl php8.2-zip php8.2-gd

# Install Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer

# Install MySQL
sudo apt install mysql-server

# Install Redis
sudo apt install redis-server
```

### Step 1: Setup Database

```bash
# Login to MySQL
mysql -u root -p

# Create database and user
CREATE DATABASE pxebot;
CREATE USER 'laravel'@'localhost' IDENTIFIED BY 'secret';
GRANT ALL PRIVILEGES ON pxebot.* TO 'laravel'@'localhost';
FLUSH PRIVILEGES;
EXIT;
```

### Step 2: Configure Environment

```bash
cd /Users/cyb3rdog/Downloads/pxebots

# Copy .env
cp .env.example .env

# Edit .env
nano .env
```

**Update these values:**
```env
APP_URL=http://localhost:8000

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=pxebot
DB_USERNAME=laravel
DB_PASSWORD=secret

REDIS_HOST=127.0.0.1

QUEUE_CONNECTION=redis

# Add your Google and OpenAI credentials
YT_CLIENT_ID=...
YT_CLIENT_SECRET=...
OPENAI_API_KEY=...
```

### Step 3: Install Dependencies

```bash
# Install PHP dependencies
composer install

# Generate app key
php artisan key:generate

# Run migrations
php artisan migrate

# Link storage
php artisan storage:link
```

### Step 4: Start Services

**Terminal 1: Web Server**
```bash
php artisan serve
# App will be at http://localhost:8000
```

**Terminal 2: Queue Worker (Horizon)**
```bash
php artisan horizon
```

**Terminal 3: Scheduler**
```bash
php artisan schedule:work
```

### Step 5: Access Application

- **App**: http://localhost:8000
- **Horizon**: http://localhost:8000/horizon

---

## Google Cloud Console Setup

### 1. Create Project
1. Go to https://console.cloud.google.com/
2. Click "New Project"
3. Name: "PXEBot"
4. Click "Create"

### 2. Enable YouTube Data API v3
1. Go to "APIs & Services" → "Library"
2. Search "YouTube Data API v3"
3. Click "Enable"

### 3. Create OAuth2 Credentials
1. Go to "APIs & Services" → "Credentials"
2. Click "+ CREATE CREDENTIALS" → "OAuth client ID"
3. Application type: "Web application"
4. Name: "PXEBot"
5. Authorized redirect URIs:
   - `http://localhost/oauth/google/callback` (Docker)
   - `http://localhost:8000/oauth/google/callback` (Local)
6. Click "Create"
7. Copy "Client ID" and "Client Secret" to `.env`

### 4. Configure OAuth Consent Screen
1. Go to "OAuth consent screen"
2. User Type: "External"
3. Fill in:
   - App name: PXEBot
   - User support email: your email
   - Developer contact: your email
4. Add scopes:
   - `https://www.googleapis.com/auth/youtube.readonly`
   - `https://www.googleapis.com/auth/youtube.force-ssl`
   - `https://www.googleapis.com/auth/youtube`
5. Add test users (your YouTube account email)
6. Save

---

## OpenAI Setup

1. Go to https://platform.openai.com/api-keys
2. Click "+ Create new secret key"
3. Name: "PXEBot"
4. Copy key to `.env` as `OPENAI_API_KEY`

**Recommended model:**
- `gpt-4o-mini` (cheap, fast) - $0.15/1M input tokens
- `gpt-4o` (better quality) - $2.50/1M input tokens

---

## First Time Usage

### 1. Register Account
1. Go to http://localhost (or http://localhost:8000)
2. Click "Register"
3. Create account

### 2. Connect YouTube Channel
1. Login to dashboard
2. Click "Connect YouTube Channel"
3. Authorize with Google
4. Select your YouTube channel

### 3. Start Bot
1. In dashboard, click "Start Bot"
2. Bot status should show "RUNNING"

### 4. Create Your First Command
1. Click "Commands"
2. Click "+ Add Command"
3. Example:
   - Name: `discord`
   - Response: `Join our Discord: https://discord.gg/yourserver`
   - Cooldown: `10`
   - Role: `viewer`
   - Enabled: ✅
4. Click "Save"

### 5. Test in YouTube Live Chat
1. Start a YouTube live stream
2. In chat, type: `!discord`
3. Bot should reply!

---

## Troubleshooting

### Docker Issues

**"Cannot connect to Docker daemon"**
- Make sure Docker Desktop is running
- Restart Docker Desktop
- Check Docker Desktop settings

**"Port already in use"**
```bash
# Change port in .env
APP_PORT=8080

# Then rebuild
docker-compose down
docker-compose up -d
```

**"Permission denied"**
```bash
# Fix permissions
docker-compose exec app chown -R www-data:www-data storage bootstrap/cache
docker-compose exec app chmod -R 775 storage bootstrap/cache
```

### Local Setup Issues

**"Class not found"**
```bash
composer dump-autoload
php artisan config:clear
```

**"Database connection failed"**
- Check MySQL is running: `mysql -u laravel -psecret`
- Verify credentials in `.env`

**"Redis connection refused"**
- macOS: `brew services start redis`
- Linux: `sudo systemctl start redis`
- Windows: Start Redis in Services

**"Queue not processing"**
- Make sure Horizon is running: `php artisan horizon`
- Check: http://localhost:8000/horizon

---

## Development Tips

### Useful Commands

```bash
# Clear caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# Run tests
php artisan test

# View logs
tail -f storage/logs/laravel.log

# Database
php artisan migrate:fresh --seed  # Reset DB with demo data
php artisan tinker  # Interactive console
```

### Hot Reload (Optional)

Install npm dependencies and run Vite:
```bash
npm install
npm run dev
```

---

## Production Deployment

See `DEPLOYMENT.md` for production setup on:
- VPS (DigitalOcean, Linode, AWS)
- Shared hosting
- Managed Laravel hosting (Laravel Forge, Ploi)

---

## Need Help?

1. Check logs:
   - Docker: `docker-compose logs -f app`
   - Local: `tail -f storage/logs/laravel.log`

2. Common issues in `DEPLOYMENT.md`

3. Database issues:
   - Verify MySQL is running
   - Check credentials in `.env`
   - Run migrations again: `php artisan migrate`

4. Queue issues:
   - Restart Horizon: `docker-compose restart horizon`
   - Or: `php artisan horizon:terminate && php artisan horizon`

---

## Quick Reference

### Docker Commands
```bash
docker-compose up -d          # Start
docker-compose down           # Stop
docker-compose restart        # Restart
docker-compose logs -f app    # View logs
docker-compose exec app bash  # Shell access
```

### Artisan Commands
```bash
php artisan migrate           # Run migrations
php artisan db:seed          # Seed database
php artisan tinker           # Console
php artisan queue:work       # Queue worker
php artisan horizon          # Horizon
php artisan schedule:work    # Scheduler
```

---

**You're all set! 🚀**

Choose Docker (easier) or Local setup (more control) and follow the steps above.
