<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Http\Requests\Command\StoreCommandRequest;
use App\Http\Requests\Command\UpdateCommandRequest;
use App\Models\Audit;
use App\Models\Command;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class CommandController extends Controller
{
    public function index(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $commands = $tenant->commands()
            ->orderBy('name')
            ->paginate(20);

        return view('tenant.commands.index', compact('tenant', 'commands'));
    }

    public function create(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        return view('tenant.commands.create', compact('tenant'));
    }

    public function store(StoreCommandRequest $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $command = $tenant->commands()->create($request->validated());

        Audit::log($tenant->id, $request->user()->id, 'command_created', [
            'command_id' => $command->id,
            'name' => $command->name,
        ]);

        return redirect()->route('tenant.commands.index', $tenant)
            ->with('success', 'Command created successfully!');
    }

    public function edit(Request $request, Tenant $tenant, Command $command)
    {
        Gate::authorize('manage', $tenant);

        if ($command->tenant_id !== $tenant->id) {
            abort(404);
        }

        return view('tenant.commands.edit', compact('tenant', 'command'));
    }

    public function update(UpdateCommandRequest $request, Tenant $tenant, Command $command)
    {
        Gate::authorize('manage', $tenant);

        if ($command->tenant_id !== $tenant->id) {
            abort(404);
        }

        $command->update($request->validated());

        Audit::log($tenant->id, $request->user()->id, 'command_updated', [
            'command_id' => $command->id,
            'name' => $command->name,
        ]);

        return redirect()->route('tenant.commands.index', $tenant)
            ->with('success', 'Command updated successfully!');
    }

    public function destroy(Request $request, Tenant $tenant, Command $command)
    {
        Gate::authorize('manage', $tenant);

        if ($command->tenant_id !== $tenant->id) {
            abort(404);
        }

        $commandName = $command->name;
        $command->delete();

        Audit::log($tenant->id, $request->user()->id, 'command_deleted', [
            'name' => $commandName,
        ]);

        return redirect()->route('tenant.commands.index', $tenant)
            ->with('success', 'Command deleted successfully.');
    }
}
