<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Models\Audit;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class TenantController extends Controller
{
    /**
     * Start bot for tenant
     */
    public function start(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        if (!$tenant->oauthTokens()->exists()) {
            return back()->with('error', 'Please connect your YouTube channel first.');
        }

        $tenant->update([
            'status' => 'running',
            'last_error' => null,
            'last_error_at' => null,
        ]);

        Audit::log($tenant->id, $request->user()->id, 'bot_started');

        return back()->with('success', 'Bot started successfully!');
    }

    /**
     * Stop bot for tenant
     */
    public function stop(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $tenant->update(['status' => 'stopped']);

        Audit::log($tenant->id, $request->user()->id, 'bot_stopped');

        return back()->with('success', 'Bot stopped successfully.');
    }

    /**
     * Delete tenant
     */
    public function destroy(Request $request, Tenant $tenant)
    {
        Gate::authorize('manage', $tenant);

        $tenant->update(['status' => 'stopped']);
        $tenant->delete();

        Audit::log(null, $request->user()->id, 'tenant_deleted', [
            'tenant_id' => $tenant->id,
            'channel_id' => $tenant->channel_id,
        ]);

        return redirect()->route('dashboard')
            ->with('success', 'Channel disconnected successfully.');
    }

    /**
     * Toggle a boolean setting
     */
    public function toggleSetting(Request $request, Tenant $tenant, string $key)
    {
        Gate::authorize('manage', $tenant);

        // Validate setting key
        $allowedSettings = ['roasting_mode'];

        if (!in_array($key, $allowedSettings)) {
            return response()->json(['error' => 'Invalid setting key'], 400);
        }

        // Get current value
        $setting = $tenant->settings()->firstOrNew(['key' => $key]);
        $currentValue = $setting->value ?? '0';

        // Toggle value
        $newValue = $currentValue === '1' ? '0' : '1';

        $tenant->settings()->updateOrCreate(
            ['key' => $key],
            ['value' => $newValue]
        );

        // Log the action
        Audit::log($tenant->id, $request->user()->id, 'setting_toggled', [
            'setting' => $key,
            'new_value' => $newValue,
        ]);

        return response()->json([
            'success' => true,
            'enabled' => $newValue === '1',
            'message' => ucfirst(str_replace('_', ' ', $key)) . ' ' . ($newValue === '1' ? 'enabled' : 'disabled'),
        ]);
    }
}
