<?php

namespace App\Jobs;

use App\Models\Filter;
use App\Models\GiveawayEntry;
use App\Models\MessageLog;
use App\Models\Tenant;
use App\Services\CommandRuntime;
use App\Services\OpenAIService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessIncomingMessage implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 2;
    public int $timeout = 60;

    public function __construct(
        public Tenant $tenant,
        public array $message
    ) {}

    public function handle(
        CommandRuntime $commandRuntime,
        OpenAIService $openai
    ): void {
        $snippet = $this->message['snippet'];
        $author = $this->message['authorDetails'];

        $messageId = $this->message['id'];
        $text = $snippet['textMessageDetails']['messageText'] ?? '';
        $authorChannelId = $author['channelId'];
        $authorName = $author['displayName'];

        // Create message log entry
        $messageLog = MessageLog::create([
            'tenant_id' => $this->tenant->id,
            'msg_id' => $messageId,
            'author_channel_id' => $authorChannelId,
            'author_name' => $authorName,
            'text' => $text,
            'created_at' => now(),
        ]);

        // Check filters first
        $filterResult = $this->checkFilters($text);
        if ($filterResult) {
            $messageLog->update([
                'handled_by' => 'filter',
                'posted_reply' => $filterResult,
            ]);

            // Post filter warning if action is 'warn'
            if ($filterResult !== 'ignored') {
                PostBotReply::dispatch($this->tenant, $filterResult);
            }

            return;
        }

        // Check for giveaway entry
        if ($this->handleGiveawayEntry($text, $authorChannelId, $authorName)) {
            $messageLog->update(['handled_by' => 'command']);
            return;
        }

        // Try command execution
        $context = [
            'tenant_id' => $this->tenant->id,
            'author_id' => $authorChannelId,
            'author_name' => $authorName,
            'channel_name' => $this->tenant->channel_title,
            'uptime' => $this->tenant->getUptime() ?? 0,
        ];

        $commandResponse = $commandRuntime->execute($this->tenant, $text, $context);

        if ($commandResponse) {
            $messageLog->update([
                'handled_by' => 'command',
                'posted_reply' => $commandResponse,
            ]);

            PostBotReply::dispatch($this->tenant, $commandResponse);
            return;
        }

        // Classify message
        $classification = $openai->classify($text);
        $messageLog->update([
            'detected_lang' => $classification['language'],
            'classification' => $classification,
        ]);

        // Check if we should respond with OpenAI
        if (!$openai->shouldRespond($text, $classification)) {
            $messageLog->update(['handled_by' => 'ignored']);
            return;
        }

        // Check content moderation if enabled
        if (config('services.openai.enable_moderation', true)) {
            $moderation = $openai->moderate($text);
            if ($moderation['flagged']) {
                $messageLog->update([
                    'handled_by' => 'filter',
                    'error' => ['moderation' => 'Content flagged by moderation'],
                ]);
                return;
            }
        }

        // Generate OpenAI response
        $reply = $openai->respond(
            $this->tenant,
            $text,
            $classification['language'],
            $classification['script'] ?? 'latin',
            $classification
        );

        if ($reply) {
            $messageLog->update([
                'handled_by' => 'openai',
                'posted_reply' => $reply,
            ]);

            PostBotReply::dispatch($this->tenant, $reply);
        } else {
            $messageLog->update([
                'handled_by' => 'ignored',
                'error' => ['openai' => 'Failed to generate response'],
            ]);
        }
    }

    /**
     * Check all filters
     */
    private function checkFilters(string $text): ?string
    {
        $filters = $this->tenant->filters()->where('is_enabled', true)->get();

        foreach ($filters as $filter) {
            if ($filter->check($text)) {
                return match ($filter->action) {
                    'warn' => "⚠️ Please avoid {$filter->type} in this chat.",
                    'timeout' => "Message filtered: {$filter->type}",
                    'ignore' => 'ignored',
                    default => null,
                };
            }
        }

        return null;
    }

    /**
     * Handle giveaway entry
     */
    private function handleGiveawayEntry(string $text, string $channelId, string $name): bool
    {
        $activeGiveaway = $this->tenant->giveaways()
            ->where('status', 'active')
            ->first();

        if (!$activeGiveaway) {
            return false;
        }

        $keyword = strtolower($activeGiveaway->keyword);
        $textLower = strtolower(trim($text));

        if ($textLower === $keyword || str_starts_with($textLower, $keyword . ' ')) {
            GiveawayEntry::firstOrCreate([
                'giveaway_id' => $activeGiveaway->id,
                'user_channel_id' => $channelId,
            ], [
                'user_name' => $name,
                'entered_at' => now(),
            ]);

            return true;
        }

        return false;
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('ProcessIncomingMessage job failed', [
            'tenant_id' => $this->tenant->id,
            'message_id' => $this->message['id'] ?? 'unknown',
            'error' => $exception->getMessage(),
        ]);
    }
}
