<?php

namespace App\Services;

use App\Models\Tenant;
use Illuminate\Support\Facades\Redis;

class RateLimiterService
{
    private const OUTBOUND_PER_MIN_KEY = 'rate_limit:outbound:min:';
    private const OUTBOUND_PER_HOUR_KEY = 'rate_limit:outbound:hour:';

    /**
     * Check if tenant can send outbound message
     */
    public function allowOutbound(Tenant $tenant): bool
    {
        $perMinLimit = config('services.bot.rate_limit_outbound_per_min', 12);
        $perHourLimit = config('services.bot.rate_limit_outbound_per_hour', 200);

        // Check per-minute limit
        $minKey = self::OUTBOUND_PER_MIN_KEY . $tenant->id;
        $minCount = (int) Redis::get($minKey);

        if ($minCount >= $perMinLimit) {
            return false;
        }

        // Check per-hour limit
        $hourKey = self::OUTBOUND_PER_HOUR_KEY . $tenant->id;
        $hourCount = (int) Redis::get($hourKey);

        if ($hourCount >= $perHourLimit) {
            return false;
        }

        return true;
    }

    /**
     * Record outbound message
     */
    public function recordOutbound(Tenant $tenant): void
    {
        $minKey = self::OUTBOUND_PER_MIN_KEY . $tenant->id;
        $hourKey = self::OUTBOUND_PER_HOUR_KEY . $tenant->id;

        // Increment per-minute counter
        Redis::incr($minKey);
        if (!Redis::ttl($minKey)) {
            Redis::expire($minKey, 60);
        }

        // Increment per-hour counter
        Redis::incr($hourKey);
        if (!Redis::ttl($hourKey)) {
            Redis::expire($hourKey, 3600);
        }
    }

    /**
     * Get remaining quota for tenant
     */
    public function getRemainingQuota(Tenant $tenant): array
    {
        $perMinLimit = config('services.bot.rate_limit_outbound_per_min', 12);
        $perHourLimit = config('services.bot.rate_limit_outbound_per_hour', 200);

        $minKey = self::OUTBOUND_PER_MIN_KEY . $tenant->id;
        $hourKey = self::OUTBOUND_PER_HOUR_KEY . $tenant->id;

        $minCount = (int) Redis::get($minKey);
        $hourCount = (int) Redis::get($hourKey);

        return [
            'per_minute' => [
                'limit' => $perMinLimit,
                'used' => $minCount,
                'remaining' => max(0, $perMinLimit - $minCount),
            ],
            'per_hour' => [
                'limit' => $perHourLimit,
                'used' => $hourCount,
                'remaining' => max(0, $perHourLimit - $hourCount),
            ],
        ];
    }

    /**
     * Reset rate limits for tenant (admin override)
     */
    public function reset(Tenant $tenant): void
    {
        $minKey = self::OUTBOUND_PER_MIN_KEY . $tenant->id;
        $hourKey = self::OUTBOUND_PER_HOUR_KEY . $tenant->id;

        Redis::del($minKey);
        Redis::del($hourKey);
    }
}
