<?php

namespace App\Services;

use App\Models\OauthToken;
use App\Models\Tenant;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class YouTubeService
{
    private const YOUTUBE_API_BASE = 'https://www.googleapis.com/youtube/v3';
    private const OAUTH_TOKEN_URL = 'https://oauth2.googleapis.com/token';

    public function __construct(
        private RateLimiterService $rateLimiter
    ) {}

    /**
     * Get or refresh access token for a tenant
     */
    public function getAccessToken(Tenant $tenant): ?string
    {
        $token = $tenant->oauthTokens()->where('provider', 'google')->first();

        if (!$token) {
            return null;
        }

        if ($token->needsRefresh()) {
            $this->refreshToken($token);
            $token->refresh();
        }

        return $token->access_token;
    }

    /**
     * Refresh OAuth token
     */
    public function refreshToken(OauthToken $token): bool
    {
        try {
            $response = Http::post(self::OAUTH_TOKEN_URL, [
                'client_id' => config('services.google.client_id'),
                'client_secret' => config('services.google.client_secret'),
                'refresh_token' => $token->refresh_token,
                'grant_type' => 'refresh_token',
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $token->update([
                    'access_token' => $data['access_token'],
                    'expires_at' => now()->addSeconds($data['expires_in']),
                ]);

                Log::info('YouTube token refreshed', ['tenant_id' => $token->tenant_id]);
                return true;
            }

            Log::error('Failed to refresh YouTube token', [
                'tenant_id' => $token->tenant_id,
                'response' => $response->json(),
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error('Exception refreshing YouTube token', [
                'tenant_id' => $token->tenant_id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Get active broadcast and liveChatId
     */
    public function getActiveBroadcast(Tenant $tenant): ?array
    {
        $accessToken = $this->getAccessToken($tenant);
        if (!$accessToken) {
            return null;
        }

        try {
            $response = Http::withToken($accessToken)->get(self::YOUTUBE_API_BASE . '/liveBroadcasts', [
                'part' => 'snippet,contentDetails,status',
                'broadcastStatus' => 'active',
                'broadcastType' => 'all',
            ]);

            if ($response->successful()) {
                $data = $response->json();
                if (isset($data['items'][0])) {
                    $broadcast = $data['items'][0];
                    return [
                        'broadcast_id' => $broadcast['id'],
                        'live_chat_id' => $broadcast['snippet']['liveChatId'] ?? null,
                        'title' => $broadcast['snippet']['title'],
                        'started_at' => $broadcast['snippet']['actualStartTime'] ?? null,
                    ];
                }
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Error fetching active broadcast', [
                'tenant_id' => $tenant->id,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Poll live chat messages
     */
    public function pollMessages(Tenant $tenant, ?string $pageToken = null): ?array
    {
        $accessToken = $this->getAccessToken($tenant);
        if (!$accessToken || !$tenant->live_chat_id) {
            return null;
        }

        try {
            $params = [
                'liveChatId' => $tenant->live_chat_id,
                'part' => 'id,snippet,authorDetails',
                'maxResults' => config('services.youtube.poll_max_results', 200),
            ];

            if ($pageToken) {
                $params['pageToken'] = $pageToken;
            }

            $response = Http::withToken($accessToken)->get(
                self::YOUTUBE_API_BASE . '/liveChat/messages',
                $params
            );

            if ($response->successful()) {
                $data = $response->json();

                return [
                    'messages' => $data['items'] ?? [],
                    'next_page_token' => $data['nextPageToken'] ?? null,
                    'polling_interval_millis' => $data['pollingIntervalMillis'] ?? 2000,
                ];
            }

            // Handle rate limits
            if ($response->status() === 429) {
                Log::warning('YouTube API rate limit hit', ['tenant_id' => $tenant->id]);
                return null;
            }

            // Handle other errors
            if ($response->status() === 403) {
                Log::error('YouTube API forbidden', [
                    'tenant_id' => $tenant->id,
                    'response' => $response->json(),
                ]);
                $tenant->update([
                    'status' => 'error',
                    'last_error' => 'API access forbidden - check OAuth scopes',
                    'last_error_at' => now(),
                ]);
                return null;
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Error polling YouTube messages', [
                'tenant_id' => $tenant->id,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Send message to live chat
     */
    public function sendMessage(Tenant $tenant, string $text): bool
    {
        $accessToken = $this->getAccessToken($tenant);
        if (!$accessToken || !$tenant->live_chat_id) {
            return false;
        }

        // Check rate limit
        if (!$this->rateLimiter->allowOutbound($tenant)) {
            Log::warning('Rate limit exceeded for tenant', ['tenant_id' => $tenant->id]);
            return false;
        }

        try {
            $response = Http::withToken($accessToken)
                ->post(self::YOUTUBE_API_BASE . '/liveChat/messages', [
                    'part' => 'snippet',
                    'snippet' => [
                        'liveChatId' => $tenant->live_chat_id,
                        'type' => 'textMessageEvent',
                        'textMessageDetails' => [
                            'messageText' => $text,
                        ],
                    ],
                ]);

            if ($response->successful()) {
                $this->rateLimiter->recordOutbound($tenant);
                Log::info('Message sent to YouTube', [
                    'tenant_id' => $tenant->id,
                    'text' => substr($text, 0, 50),
                ]);
                return true;
            }

            Log::error('Failed to send message to YouTube', [
                'tenant_id' => $tenant->id,
                'status' => $response->status(),
                'response' => $response->json(),
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error('Exception sending message to YouTube', [
                'tenant_id' => $tenant->id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Get channel info
     */
    public function getChannelInfo(string $accessToken): ?array
    {
        try {
            $response = Http::withToken($accessToken)->get(self::YOUTUBE_API_BASE . '/channels', [
                'part' => 'id,snippet,statistics',
                'mine' => 'true',
            ]);

            if ($response->successful()) {
                $data = $response->json();
                if (isset($data['items'][0])) {
                    $channel = $data['items'][0];
                    return [
                        'channel_id' => $channel['id'],
                        'title' => $channel['snippet']['title'],
                        'handle' => $channel['snippet']['customUrl'] ?? null,
                        'subscribers' => $channel['statistics']['subscriberCount'] ?? 0,
                    ];
                }
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Error fetching channel info', ['error' => $e->getMessage()]);
            return null;
        }
    }
}
